import { faUndo } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button } from '@yandex-cloud/uikit';
import {
   EMPTY_VALUE,
   EnumField2,
   FieldLayout2,
   FormCard,
   IEnumOption,
   InputField2,
   isEqual,
   RemoveButton,
   WarningPlate,
} from '@yandex-infracloud-ui/libs';
import { getIn, useFormikContext } from 'formik';
import React, { useCallback, useEffect, useMemo, useState } from 'react';
import {
   DiskStaticResource,
   StaticResourceFileType,
   StaticResourceType,
   unmodifiedAccessPermissionsLabel,
} from '../../../../../models/ui';

import { EResourceAccessPermissions } from '../../../../../proto-typings';
import { EnabledSwitcherField } from '../../../../forms';
import { SandboxUrlField } from '../../../_common/SanboxUrlField/SandboxUrlField';
import { AccessPermissionsHint } from '../../hints';

import classes from './DiskStaticResourceCard.module.css';
import { FilesSubForm } from './Files/FilesSubForm';

const resourceTypeOptions: IEnumOption[] = [
   { title: StaticResourceType.Url, value: StaticResourceType.Url },
   { title: StaticResourceType.Files, value: StaticResourceType.Files },
];

interface Props {
   disabled: boolean;
   index: number;
   name: string;
   readonly: boolean;
   soxService: boolean;

   onRemove(i: number): void;

   onRestore(i: number): void;
}

const AccessPermissionsHelp: React.FC<{
   accessPermissions: EResourceAccessPermissions;
}> = React.memo(({ accessPermissions }) => (
   <>
      {`User: read/write | `}

      {`Group: ${
         accessPermissions === EResourceAccessPermissions.EResourceAccessPermissions_600 ? 'none' : 'read/write'
      } | `}

      {`Other: ${
         accessPermissions === EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED ? 'read/write' : 'none'
      }`}
   </>
));

export const DiskStaticResourceCard: React.FC<Props> = React.memo(
   ({ name, index, readonly, disabled, soxService, onRemove, onRestore }) => {
      const form = useFormikContext();
      const value: DiskStaticResource = getIn(form.values, name);
      const initialValue: DiskStaticResource = getIn(form.initialValues, name);
      const touched = getIn(form.touched, `${name}`);

      const [isAllow600, setIsAllow600] = useState(
         value.accessPermissions === EResourceAccessPermissions.EResourceAccessPermissions_600,
      );

      const resourceAccessPermissionsOptions = useMemo(
         () =>
            [
               EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED,
               EResourceAccessPermissions.EResourceAccessPermissions_600,
               EResourceAccessPermissions.EResourceAccessPermissions_660,
            ]
               .filter(
                  v =>
                     // не даём выбирать EResourceAccessPermissions_600 через UI
                     // показываем только для продвинутых пользователей, когда он уже есть в спеке
                     v !== EResourceAccessPermissions.EResourceAccessPermissions_600 || isAllow600,
               )
               .map(v => ({
                  title:
                     v === EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED
                        ? unmodifiedAccessPermissionsLabel
                        : v,
                  value: v,
               })),
         [isAllow600],
      );

      const showAccessPermissionsWarning = useMemo(() => {
         if (soxService) {
            const hasSecretFile =
               value.type === StaticResourceType.Files &&
               value.files?.some(v => v.type === StaticResourceFileType.Secret);

            if (
               hasSecretFile &&
               value.accessPermissions === EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED
            ) {
               return true;
            }
         }

         return false;
      }, [soxService, value]);

      useEffect(() => {
         if (!isAllow600 && value.accessPermissions === EResourceAccessPermissions.EResourceAccessPermissions_600) {
            setIsAllow600(true);
         }
      }, [value.accessPermissions, isAllow600, setIsAllow600]);

      useEffect(() => {
         if (!isEqual(value, initialValue) && (!touched?.id || !touched?.mountPoint)) {
            if (!touched?.id) {
               form.setFieldTouched(`${name}.id`, true);
            }

            if (!touched?.mountPoint) {
               form.setFieldTouched(`${name}.mountPoint`, true);
            }
         }
      }, [form, name, initialValue, value, touched]);

      const handleRemoveStaticResource = useCallback(() => onRemove(index), [index, onRemove]);
      const handleRestoreStaticResource = useCallback(() => onRestore(index), [index, onRestore]);

      return (
         <div data-test={`StaticResource:${index + 1}`}>
            <FormCard
               title={value.id || EMPTY_VALUE}
               className={value.removed ? classes.removed : undefined}
               buttons={
                  !readonly && !disabled ? (
                     <>
                        {value.removed ? (
                           <div data-e2e={'StaticResource:Restore'}>
                              <Button view={'flat'} title={'Restore'} onClick={handleRestoreStaticResource}>
                                 <FontAwesomeIcon icon={faUndo} />
                              </Button>
                           </div>
                        ) : (
                           <div data-e2e={'StaticResource:Remove'}>
                              <RemoveButton onClick={handleRemoveStaticResource} size={'lg'} />
                           </div>
                        )}
                     </>
                  ) : null
               }
            >
               {value.removed === true ? (
                  <FieldLayout2 name={`${name}.removed`} label={null}>
                     removed
                  </FieldLayout2>
               ) : (
                  <>
                     {value.type === StaticResourceType.Unknown ? (
                        <div className={classes.unknown}>Unknown (for UI) static resource type</div>
                     ) : (
                        <div className={classes.form}>
                           <InputField2
                              name={`${name}.id`}
                              label={'ID'}
                              disabled={disabled}
                              readonly={readonly}
                              readonlyDots={readonly}
                           />

                           <EnumField2
                              name={`${name}.type`}
                              label={'Type'}
                              // eslint-disable-next-line no-underscore-dangle
                              disabled={disabled || value._order !== undefined}
                              readonly={readonly}
                              controlProps={{ options: resourceTypeOptions }}
                              readonlyDots={readonly}
                           />

                           {value.type === StaticResourceType.Url ? (
                              <SandboxUrlField
                                 name={`${name}.url`}
                                 label={'URL'}
                                 disabled={disabled}
                                 readonly={readonly}
                                 readonlyDots={readonly}
                              />
                           ) : null}

                           {value.type === StaticResourceType.Files ? (
                              <FilesSubForm name={`${name}.files`} readonly={readonly} />
                           ) : null}

                           <EnumField2
                              name={`${name}.accessPermissions`}
                              label={'Access permissions'}
                              hint={<AccessPermissionsHint />}
                              help={<AccessPermissionsHelp accessPermissions={value.accessPermissions} />}
                              disabled={disabled}
                              readonly={readonly}
                              controlProps={{
                                 options: resourceAccessPermissionsOptions,
                              }}
                              readonlyDots={readonly}
                           />

                           {showAccessPermissionsWarning ? (
                              <WarningPlate className={classes.warning}>
                                 Secret files should not have '{unmodifiedAccessPermissionsLabel}' access permissions
                              </WarningPlate>
                           ) : null}

                           <EnabledSwitcherField
                              name={`${name}.verification.enabled`}
                              label={'Verification'}
                              disabled={disabled}
                              readonly={readonly}
                              bigLabel={!readonly}
                              readonlyDots={readonly}
                           />

                           <InputField2
                              name={`${name}.verification.checksum`}
                              label={'Checksum'}
                              disabled={disabled}
                              readonly={readonly}
                              readonlyDots={readonly}
                           />
                        </div>
                     )}
                  </>
               )}
            </FormCard>
         </div>
      );
   },
);

DiskStaticResourceCard.displayName = 'DiskStaticResourceCard';
