import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { EMPTY_VALUE, FieldLayout2, FormButton } from '@yandex-infracloud-ui/libs';
import React, { useCallback } from 'react';
import { useFormikContext } from 'formik';

import { EResourceAccessPermissions } from '../../../../../proto-typings';
import { DiskStaticResource, StaticResourceType } from '../../../../../models/ui';
import { getHexRef } from '../../../../../utils';
import { DiskStaticResourcesHint } from '../../hints';
import { DiskStaticResourceCard } from './DiskStaticResourceCard';

import classes from './DiskStaticResourcesSubForm.module.css';

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;
   soxService: boolean;
}

export const DiskStaticResourcesSubForm: React.FC<Props> = React.memo(({ disabled, name, readonly, soxService }) => {
   const form = useFormikContext();
   const { value } = form.getFieldMeta<DiskStaticResource[]>(name);

   const handleAddResource = useCallback(() => {
      const newResource: DiskStaticResource = {
         type: StaticResourceType.Url,
         id: '',
         url: '',
         verification: { enabled: false, checksum: null },
         accessPermissions: EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED,
         _ref: getHexRef(),
      };

      form.setFieldValue(name, [...value, newResource]);
   }, [form, name, value]);

   const handleRemoveStaticResource = useCallback(
      (i: number) => {
         const newValue = [...value].map((v, index) =>
            index === i
               ? {
                    ...v,
                    removed: true,
                 }
               : v,
         );

         form.setFieldTouched(`${name}[${i}].removed`, true);
         form.setFieldValue(name, newValue);
      },
      [form, name, value],
   );

   const handleRestoreStaticResource = useCallback(
      (i: number) => {
         const newValue = [...value].map((v, index) =>
            index === i
               ? {
                    ...v,
                    removed: false,
                 }
               : v,
         );

         form.setFieldTouched(`${name}[${i}].removed`, false);
         form.setFieldValue(name, newValue);
      },
      [form, name, value],
   );

   return (
      <FieldLayout2
         name={'staticResources'}
         label={'Static resources'}
         hint={<DiskStaticResourcesHint />}
         hideErrors={true}
         bigLabel={true}
      >
         <div data-test={'Disk:StaticResources'}>
            {value.length === 0 ? (
               <>{readonly ? <div className={classes.empty}>{EMPTY_VALUE}</div> : null}</>
            ) : (
               value.map((_, i) => (
                  <DiskStaticResourceCard
                     name={`${name}[${i}]`}
                     // eslint-disable-next-line react/no-array-index-key
                     key={i}
                     index={i}
                     disabled={disabled}
                     readonly={readonly}
                     onRemove={handleRemoveStaticResource}
                     onRestore={handleRestoreStaticResource}
                     soxService={soxService}
                  />
               ))
            )}

            {disabled || readonly ? null : (
               <FormButton
                  icon={faPlus}
                  onClick={handleAddResource}
                  skipLeftSpace={true}
                  dataE2e={'StaticResources:AddStaticResource'}
               >
                  Add static resource
               </FormButton>
            )}
         </div>
      </FieldLayout2>
   );
});

DiskStaticResourcesSubForm.displayName = 'DiskStaticResourcesSubForm';
