import { ControlGroupOption } from '@yandex-cloud/uikit';
import {
   DISMISS_REASON,
   EnumSwitcher,
   IModalProps,
   isEmpty,
   ModalLayout,
   ModalLayoutProps,
} from '@yandex-infracloud-ui/libs';

import React, { useCallback, useState } from 'react';

import { RawFile, SecretFile, StaticResourceFile, StaticResourceFileType } from '../../../../../../../models/ui';

import { FileIcon } from '../FileIcon/FileIcon';

import classes from './FileModalSubForm.module.css';
import { getRawFormParams, RawFileSubForm, rawInitialFormParams } from './RawFileSubForm';
import { getSecretFormParams, SecretFileSubForm, secretInitialFormParams } from './SecretFileSubForm';

const fileTypeOptions: ControlGroupOption[] = [
   { content: StaticResourceFileType.Raw, value: StaticResourceFileType.Raw },
   { content: StaticResourceFileType.Secret, value: StaticResourceFileType.Secret },
];

interface Props extends IModalProps<StaticResourceFile> {
   duId: string;
   isNew: boolean;
   nameList: (string | null)[];
   readonly: boolean;
   stageId: string;
   file?: StaticResourceFile;
}

export const FileModalSubForm: React.FC<Props> = React.memo(
   ({ stageId, duId, cancel, isNew, nameList, ok, readonly, file }) => {
      if (!isNew && !file) {
         throw new Error('isNew=false, so you should define `variable` prop');
      }

      const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);
      const [type, setType] = useState(file?.type ?? StaticResourceFileType.Raw);

      const [rawFormParams, setRawFormParams] = useState(() =>
         isNew || file!.type !== StaticResourceFileType.Raw ? rawInitialFormParams : getRawFormParams(file as RawFile),
      );

      const [secretFormParams, setSecretFormParams] = useState(() =>
         isNew || file!.type !== StaticResourceFileType.Secret
            ? secretInitialFormParams
            : getSecretFormParams(file as SecretFile),
      );

      const handleFileTypeChange = useCallback((v: StaticResourceFileType) => {
         // TODO MultiSecret
         if (v === StaticResourceFileType.Raw || v === StaticResourceFileType.Secret) {
            setType(v);
         }
      }, []);

      const handleOk = useCallback(() => {
         switch (isNew ? type : file!.type) {
            case StaticResourceFileType.Raw: {
               ok({
                  ...rawFormParams,
                  type: StaticResourceFileType.Raw,
               } as RawFile);

               break;
            }

            case StaticResourceFileType.Secret: {
               ok({
                  ...secretFormParams,
                  type: StaticResourceFileType.Secret,
               } as SecretFile);

               break;
            }
         }
      }, [isNew, ok, type, file, rawFormParams, secretFormParams]);

      const isOkButtonDisabled = useCallback(() => {
         switch (type) {
            case StaticResourceFileType.Raw: {
               const { name } = rawFormParams;

               if (name && !isEmpty(name)) {
                  return nameList.includes(name);
               }

               break;
            }

            case StaticResourceFileType.Secret: {
               const { name, secret } = secretFormParams;

               if (name && !isEmpty(name) && !isEmpty(secret)) {
                  if (!isEmpty(secret?.alias) && !isEmpty(secret?.key)) {
                     return nameList.includes(name);
                  }
               }
               break;
            }
         }

         return true;
      }, [nameList, type, rawFormParams, secretFormParams]);

      const modalProps: ModalLayoutProps = isNew
         ? {
              title: 'Static resource file',
              onDismiss: dismiss,
              okLabel: 'Create',
              onOk: handleOk,
              disabled: isOkButtonDisabled(),
           }
         : {
              title: file!.name,
              subtitle: <div className={classes.subtitle}>{<FileIcon type={file!.type} showText={true} />}</div>,
              onDismiss: dismiss,
              okLabel: readonly ? 'Ok' : 'Apply',
              disabled: isOkButtonDisabled(),
              onOk: handleOk,
           };

      return (
         <ModalLayout {...modalProps}>
            <div className={classes.wrapper}>
               {isNew ? (
                  <div className={classes.type}>
                     <div className={classes.typeLabel}>Type</div>

                     <EnumSwitcher
                        name={'viewType'}
                        value={type}
                        onChange={handleFileTypeChange}
                        options={fileTypeOptions}
                        readonly={readonly}
                        disabled={!isNew}
                     />
                  </div>
               ) : null}

               {type === StaticResourceFileType.Raw ? (
                  <div>
                     <RawFileSubForm value={rawFormParams} onChange={setRawFormParams} readonly={readonly} />
                  </div>
               ) : null}

               {type === StaticResourceFileType.Secret ? (
                  <SecretFileSubForm
                     duId={duId}
                     value={secretFormParams}
                     onChange={setSecretFormParams}
                     stageId={stageId}
                     readonly={readonly}
                  />
               ) : null}

               {type === StaticResourceFileType.Unknown ? <>File type is unknown for UI</> : null}
            </div>
         </ModalLayout>
      );
   },
);

FileModalSubForm.displayName = 'FileModalSubForm';
