import { FieldLayout2, FormChangeListener, InputField2, IPureValueProps, isEmpty } from '@yandex-infracloud-ui/libs';
import { Formik } from 'formik';
import React, { useCallback } from 'react';

import { SecretFile } from '../../../../../../../models/ui';
import {
   DEFAULT_MULTISECRET_FORMAT,
   extractFormat,
   isMultisecret,
   LinkToSecret,
} from '../../../../../../../models/ui/secrets';

import { SecretSubForm } from '../../../../../../../modules/secrets';
import { isSecretAliasGenerated } from '../../../../../../../modules/secrets/helpers';

import { EMultiSecretFileContentFormat } from '../../../../../../../proto-typings';

export const secretInitialFormParams = {
   name: null as string | null,
   secret: null as LinkToSecret | null,
};

type SecretFormParams = typeof secretInitialFormParams;

export function getSecretFormParams(file: SecretFile): SecretFormParams {
   return {
      name: file.name,
      secret: file.secret ?? null,
   };
}

interface Props extends IPureValueProps<SecretFormParams> {
   duId: string;
   readonly?: boolean;
   stageId: string;
}

export const SecretFileSubForm: React.FC<Props> = React.memo(({ stageId, duId, value, onChange, readonly = false }) => {
   const handleChange = useCallback(
      (v: SecretFormParams) => {
         if (isEmpty(v.name) && v.secret) {
            v.name = generateMultisecretFileName(v.secret.key, v.secret.alias);
         }

         onChange(v);
      },
      [onChange],
   );

   return (
      <Formik initialValues={value} onSubmit={handleChange}>
         {() => (
            <>
               <FormChangeListener onChange={handleChange} />

               <InputField2
                  name={'name'}
                  label={'Name'}
                  placeholder={'Enter file name...'}
                  required={!readonly}
                  readonly={readonly}
                  readonlyDots={readonly}
               />

               <FieldLayout2
                  name={'secret'}
                  label={'Secret'}
                  required={!readonly}
                  readonly={readonly}
                  readonlyDots={readonly}
               >
                  <SecretSubForm
                     duId={duId}
                     name={'secret'}
                     stageId={stageId}
                     layout={'vertical'}
                     disabled={false}
                     readonly={readonly}
                     allowMultisecret={true}
                  />
               </FieldLayout2>
            </>
         )}
      </Formik>
   );
});

SecretFileSubForm.displayName = 'SecretFileSubForm';

/**
 * Расширения файлов по-умолчанию для форматов
 *
 * Для формата java нужно особенное расширение, поэтому создал эту карту
 */
const formatExtensions: Record<EMultiSecretFileContentFormat, string> = {
   [EMultiSecretFileContentFormat.EMultiSecretFileContentFormat_JAVA]: '.properties',
   [EMultiSecretFileContentFormat.EMultiSecretFileContentFormat_JSON]: '.json',
   [EMultiSecretFileContentFormat.EMultiSecretFileContentFormat_YAML]: '.yaml',
};

function generateMultisecretFileName(key: string, alias: string) {
   if (!isMultisecret(key)) {
      return key;
   }

   const extension = formatExtensions[extractFormat(key) ?? DEFAULT_MULTISECRET_FORMAT];

   return isSecretAliasGenerated(alias) ? `secret${extension}` : `${alias}${extension}`;
}
