import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import {
   modalService,
   classNames,
   FieldLayout2,
   isEmpty,
   sortHandler,
   FormErrors,
   FormButton,
} from '@yandex-infracloud-ui/libs';
import React, { useCallback, useMemo } from 'react';
import { useFormikContext } from 'formik';

import { noop } from '../../../../../../utils';
import tableClasses from '../../../../../../design/simpleTable.module.css';
import { StaticResourceFile } from '../../../../../../models/ui';
import { useSecretsContext } from '../../../../../../modules/secrets';

import { FileRowSubForm } from './FileRowSubForm';
import { FileModalSubForm } from './FileModal/FileModalSubForm';

import classes from './FilesSubForm.module.css';

interface Props {
   name: string;
   readonly: boolean;
}

export const FilesSubForm: React.FC<Props> = React.memo(({ name, readonly }) => {
   const { duId, stageId, reloadUsages } = useSecretsContext();
   const form = useFormikContext();
   const { value } = form.getFieldMeta<StaticResourceFile[]>(name);

   const openAddModal = useCallback(() => {
      modalService
         .open(
            FileModalSubForm,
            {
               duId,
               isNew: true,
               nameList: value?.map(v => v.name) ?? [],
               readonly: false,
               stageId,
            },
            { closeOnOutsideClick: false, dialogProps: { size: 'm' } },
         )
         .subscribe(v => {
            const { touched } = form.getFieldMeta(name);
            if (!touched) {
               form.setFieldTouched(name, true);
            }

            form.setFieldValue(name, [...(value ?? []), v]);
            reloadUsages();
         }, noop);
   }, [duId, value, stageId, form, name, reloadUsages]);

   /* eslint-disable no-underscore-dangle */
   const openEditModal = useCallback(
      (fileName: string, order: number | undefined) => {
         const file = value?.find(v => v._order === order && v.name === fileName);

         modalService
            .open(
               FileModalSubForm,
               {
                  duId,
                  isNew: false,
                  nameList: value.filter(v => v.name !== fileName).map(v => v.name),
                  readonly,
                  stageId,
                  file,
               },
               { closeOnOutsideClick: readonly, dialogProps: { size: 'm' } },
            )
            .subscribe(v => {
               const { touched } = form.getFieldMeta(name);
               if (!touched) {
                  form.setFieldTouched(name, true);
               }

               form.setFieldValue(
                  name,
                  value.map(f => {
                     if (f === file) {
                        return order !== undefined ? { _order: file._order, ...v } : v;
                     }

                     return f;
                  }),
               );

               reloadUsages();
            }, noop);
      },
      [value, duId, readonly, stageId, form, name, reloadUsages],
   );

   const handleRemove = useCallback(
      (fileName: string, order: number | undefined) => {
         const { touched } = form.getFieldMeta(name);
         if (!touched) {
            form.setFieldTouched(name, true);
         }

         form.setFieldValue(
            name,
            value.filter(v => !(v.name === fileName && v._order === order)),
         );

         reloadUsages();
      },

      [form, name, value, reloadUsages],
   );
   /* eslint-enable no-underscore-dangle */

   const sortedFileList = useMemo(() => {
      if (!value || isEmpty(value)) {
         return value;
      }

      return Array.from(value).sort((a, b) => sortHandler(a.name?.toLowerCase(), b.name?.toLowerCase()));
   }, [value]);

   return (
      <FieldLayout2 name={name} label={null} readonly={readonly} hideErrors={true}>
         {value && !isEmpty(value) ? (
            <div className={classes.files}>
               <table
                  className={classNames(tableClasses.simpleTable, classes.table, {
                     [classes.readonly]: readonly,
                  })}
               >
                  <thead>
                     <tr>
                        <th>Name</th>
                        <th>Source</th>
                        <th aria-label={'Copy'} />
                        {!readonly && <th aria-label={'Actions'} />}
                     </tr>
                  </thead>
                  <tbody>
                     {sortedFileList?.map((v, i) => (
                        <FileRowSubForm
                           // для "нехороших" стейджей с дублями файлов
                           // eslint-disable-next-line no-underscore-dangle
                           key={`${v.name}${v._order}`}
                           value={v}
                           onSelect={openEditModal}
                           onRemove={handleRemove}
                           readonly={readonly}
                        />
                     ))}
                  </tbody>
               </table>
            </div>
         ) : null}

         <FormErrors names={[name]} />

         {readonly ? null : (
            <FormButton
               icon={faPlus}
               onClick={openAddModal}
               skipLeftSpace={true}
               dataE2e={'StaticResources:AddStaticResource'}
            >
               Add file
            </FormButton>
         )}
      </FieldLayout2>
   );
});

FilesSubForm.displayName = 'FilesSubForm';
