import { faPlus, faUndo } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button, Table, TableColumnConfig } from '@yandex-cloud/uikit';
import { YCSelectItem } from '@yandex-data-ui/common';
import {
   EMPTY_VALUE,
   EnumField2,
   FieldLayout2,
   FormButton,
   FormCard,
   IEnumOption,
   InputField2,
   isEmpty,
   RemoveButton,
} from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { ReactNode, useCallback, useMemo } from 'react';

import { DiskLayer, DiskStaticResource, DiskVolume, VolumeLayer, VolumeStaticResource } from '../../../../../models/ui';
import { EVolumePersistenceType } from '../../../../../proto-typings';
import { getHexRef } from '../../../../../utils';
import { YCSelectField } from '../../../../forms';

import classes from './DiskVolumeCard.module.css';

const persistenceTypeOptions: IEnumOption[] = [
   { title: 'persistent', value: EVolumePersistenceType.EVolumePersistenceType_PERSISTENT },
   { title: 'non persistent', value: EVolumePersistenceType.EVolumePersistenceType_NON_PERSISTENT },
];

interface TableRow {
   buttons: ReactNode;
   id: ReactNode;
   // type: string;
   volumeRelativeMountPoint: ReactNode;
}

const columns: TableColumnConfig<TableRow>[] = [
   { id: 'id', name: 'Static Resource ID' },
   // { id: 'type', name: 'Type' },
   { id: 'volumeRelativeMountPoint', name: 'Volume relative mount point directory' },
   { id: 'buttons', name: '' },
];

interface Props {
   name: string;
   readonly: boolean;
   disabled: boolean;
   index: number;

   layers: DiskLayer[];
   staticResources: DiskStaticResource[];

   onRemove: (i: number) => void;

   onRestore(i: number): void;
}

export const DiskVolumeCard: React.FC<Props> = React.memo(
   ({ name, readonly, disabled, index, layers, staticResources, onRemove, onRestore }) => {
      const form = useFormikContext();
      const { value } = form.getFieldMeta<DiskVolume>(name);

      const LayerSelectOptions: YCSelectItem[] = useMemo(() => {
         if (!layers || isEmpty(layers)) {
            return [];
         }

         return layers.map(layer => ({
            // eslint-disable-next-line no-underscore-dangle
            value: layer._ref,
            title: layer.id || EMPTY_VALUE,
         }));
      }, [layers]);

      const StaticResourceSelectOptions: YCSelectItem[] = useMemo(() => {
         if (!staticResources || isEmpty(staticResources)) {
            return [];
         }

         return staticResources.map(staticResource => ({
            // eslint-disable-next-line no-underscore-dangle
            value: staticResource._ref,
            title: staticResource.id || EMPTY_VALUE,
         }));
      }, [staticResources]);

      const handleAddLayer = useCallback(() => {
         const newLayer: VolumeLayer = {
            _layerRef: getHexRef(),
         };

         form.setFieldValue(`${name}.layers`, [...value.layers, newLayer]);
      }, [form, name, value]);

      const handleRemoveLayer = useCallback(
         (i: number) => {
            const newValue = [...value.layers];

            newValue.splice(i, 1);

            form.setFieldTouched(`${name}.layers[${i}]`, false);
            form.setFieldValue(`${name}.layers`, newValue);
         },
         [form, name, value],
      );

      const handleAddStaticResource = useCallback(() => {
         const newStaticResource: VolumeStaticResource = {
            _staticResourceRef: getHexRef(),
            volumeRelativeMountPoint: null,
         };

         form.setFieldValue(`${name}.staticResources`, [...value.staticResources, newStaticResource]);
      }, [form, name, value]);

      const handleRemoveStaticResource = useCallback(
         (i: number) => {
            const newValue = [...value.staticResources];

            newValue.splice(i, 1);

            form.setFieldTouched(`${name}.staticResources[${i}]`, false);
            form.setFieldValue(`${name}.staticResources`, newValue);
         },
         [form, name, value],
      );

      const getReadonlyLayerId = useCallback(
         // eslint-disable-next-line no-underscore-dangle
         (ref: string | null) => layers.find(v => v._ref === ref)?.id || EMPTY_VALUE,
         [layers],
      );

      const getReadonlyStaticResourceId = useCallback(
         // eslint-disable-next-line no-underscore-dangle
         (ref: string | null) => staticResources.find(v => v._ref === ref)?.id || EMPTY_VALUE,
         [staticResources],
      );

      return (
         <div data-test={`Volume:${index + 1}`}>
            <FormCard
               title={value.id || EMPTY_VALUE}
               className={value.removed ? classes.removed : undefined}
               buttons={
                  !readonly && !disabled ? (
                     <>
                        {value.removed ? (
                           <div data-e2e={'Volume:Restore'}>
                              <Button view={'flat'} title={'Restore'} onClick={() => onRestore(index)}>
                                 <FontAwesomeIcon icon={faUndo} />
                              </Button>
                           </div>
                        ) : (
                           <div data-e2e={'Volume:Remove'}>
                              <RemoveButton onClick={() => onRemove(index)} size={'lg'} />
                           </div>
                        )}
                     </>
                  ) : null
               }
            >
               {value.removed === true ? (
                  <FieldLayout2 name={`${name}.removed`} label={null}>
                     removed
                  </FieldLayout2>
               ) : (
                  <>
                     <div className={classes.inputWidth}>
                        <div data-test={'Volume:Id'}>
                           <InputField2
                              name={`${name}.id`}
                              label={'Volume ID'}
                              readonly={readonly}
                              readonlyDots={readonly}
                              disabled={disabled}
                              required={true}
                           />
                        </div>
                     </div>

                     <EnumField2
                        name={`${name}.persistenceType`}
                        label={'Persistence type'}
                        // eslint-disable-next-line no-underscore-dangle
                        disabled={disabled}
                        readonly={readonly}
                        controlProps={{ options: persistenceTypeOptions }}
                        readonlyDots={readonly}
                     />

                     <div className={classes.layers}>
                        <FieldLayout2 name={`${name}.layers`} label={'Layers'} bigLabel={true} hideErrors={true}>
                           {!readonly ? (
                              <div>
                                 {isEmpty(layers)
                                    ? 'You have no layers yet. You should create disk layers first.'
                                    : null}
                              </div>
                           ) : null}

                           {isEmpty(value.layers) ? (
                              <>{readonly ? EMPTY_VALUE : null}</>
                           ) : (
                              value.layers.map((layer, i) => (
                                 // eslint-disable-next-line react/no-array-index-key
                                 <div key={i} className={classes.layer}>
                                    <div className={classes.layerSelect}>
                                       {readonly ? (
                                          // eslint-disable-next-line no-underscore-dangle
                                          getReadonlyLayerId(layer._layerRef)
                                       ) : (
                                          <YCSelectField
                                             name={`${name}.layers[${i}]._layerRef`}
                                             label={null}
                                             disabled={disabled}
                                             readonly={readonly}
                                             readonlyDots={readonly}
                                             controlProps={{
                                                items: LayerSelectOptions,
                                                placeholder: !isEmpty(LayerSelectOptions) ? 'Select layer' : undefined,
                                                showSearch: false,
                                                showItemMeta: true,
                                             }}
                                          />
                                       )}
                                    </div>

                                    {!disabled && !readonly ? (
                                       <RemoveButton onClick={() => handleRemoveLayer(i)} />
                                    ) : null}
                                 </div>
                              ))
                           )}

                           {disabled || readonly || isEmpty(layers) ? null : (
                              <FormButton
                                 icon={faPlus}
                                 onClick={handleAddLayer}
                                 skipLeftSpace={true}
                                 dataE2e={'Layers:AddLayer'}
                              >
                                 Add layer
                              </FormButton>
                           )}
                        </FieldLayout2>
                     </div>

                     <div className={classes.staticResources}>
                        <FieldLayout2
                           name={`${name}.staticResources`}
                           label={'Mount static resources'}
                           bigLabel={true}
                           hideErrors={true}
                        >
                           {!readonly ? (
                              <div>
                                 {isEmpty(staticResources)
                                    ? 'You have no static resources yet. You should create disk static resources first.'
                                    : null}
                              </div>
                           ) : null}

                           {isEmpty(value.staticResources) ? (
                              <>{readonly ? EMPTY_VALUE : null}</>
                           ) : (
                              <Table
                                 className={classes.table}
                                 columns={columns}
                                 verticalAlign={'top'}
                                 data={value.staticResources.map((staticResource, i) => ({
                                    id: (
                                       <div className={classes.staticResourceSelect}>
                                          {readonly ? (
                                             // eslint-disable-next-line no-underscore-dangle
                                             getReadonlyStaticResourceId(staticResource._staticResourceRef)
                                          ) : (
                                             <YCSelectField
                                                name={`${name}.staticResources[${i}]._staticResourceRef`}
                                                label={null}
                                                disabled={disabled}
                                                readonly={readonly}
                                                readonlyDots={readonly}
                                                controlProps={{
                                                   items: StaticResourceSelectOptions,
                                                   placeholder: !isEmpty(StaticResourceSelectOptions)
                                                      ? 'Select static resource'
                                                      : undefined,
                                                   showSearch: false,
                                                   showItemMeta: true,
                                                }}
                                             />
                                          )}
                                       </div>
                                    ),
                                    volumeRelativeMountPoint: (
                                       <div className={classes.volumeRelativeMountPoint}>
                                          {readonly ? (
                                             staticResource.volumeRelativeMountPoint || EMPTY_VALUE
                                          ) : (
                                             <InputField2
                                                name={`${name}.staticResources[${i}].volumeRelativeMountPoint`}
                                                label={null}
                                                disabled={disabled}
                                                readonly={readonly}
                                                readonlyDots={readonly}
                                                placeholder={'Volume relative mount point directory'}
                                             />
                                          )}
                                       </div>
                                    ),
                                    buttons:
                                       !disabled && !readonly ? (
                                          <RemoveButton onClick={() => handleRemoveStaticResource(i)} />
                                       ) : (
                                          ' '
                                       ),
                                 }))}
                              />
                           )}

                           {disabled || readonly || isEmpty(staticResources) ? null : (
                              <FormButton
                                 icon={faPlus}
                                 onClick={handleAddStaticResource}
                                 skipLeftSpace={true}
                                 dataE2e={'StaticResources:AddStaticResource'}
                              >
                                 Add static resource
                              </FormButton>
                           )}
                        </FieldLayout2>
                     </div>
                  </>
               )}
            </FormCard>
         </div>
      );
   },
);

DiskVolumeCard.displayName = 'DiskVolumeCard';
