import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { EMPTY_VALUE, FieldLayout2, FormButton } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback } from 'react';

import { getHexRef } from '../../../../../utils';
import { DiskLayer, DiskStaticResource, DiskVolume } from '../../../../../models/ui';
import { DiskVolumeHint } from '../../components/DisksSubForm/hints';
import { DiskVolumeCard } from './DiskVolumeCard';
import { EVolumePersistenceType } from '../../../../../proto-typings';

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;

   layers: DiskLayer[];
   staticResources: DiskStaticResource[];
}

export const DiskVolumesSubForm: React.FC<Props> = React.memo(
   ({ disabled, name, readonly, layers, staticResources }) => {
      const form = useFormikContext();
      const { value } = form.getFieldMeta<DiskVolume[]>(name);

      const handleAddVolume = useCallback(() => {
         const newVolume: DiskVolume = {
            id: '',
            layers: [],
            staticResources: [],
            persistenceType: EVolumePersistenceType.EVolumePersistenceType_PERSISTENT,
            _ref: getHexRef(),
         };

         form.setFieldValue(name, [...value, newVolume]);
      }, [form, name, value]);

      const handleRemoveVolume = useCallback(
         (i: number) => {
            const newValue = [...value].map((v, index) =>
               index === i
                  ? {
                       ...v,
                       removed: true,
                    }
                  : v,
            );

            form.setFieldTouched(`${name}[${i}].removed`, true);
            form.setFieldValue(name, newValue);
         },
         [form, name, value],
      );

      const handleRestoreVolume = useCallback(
         (i: number) => {
            const newValue = [...value].map((v, index) =>
               index === i
                  ? {
                       ...v,
                       removed: false,
                    }
                  : v,
            );

            form.setFieldTouched(`${name}[${i}].removed`, false);
            form.setFieldValue(name, newValue);
         },
         [form, name, value],
      );

      return (
         <FieldLayout2 name={name} label={'Volumes'} hint={<DiskVolumeHint />} hideErrors={true} bigLabel={true}>
            <div data-test={'Disk:Volumes'}>
               {value.length === 0 ? (
                  <>{readonly ? <div>{EMPTY_VALUE}</div> : null}</>
               ) : (
                  value.map((_, i) => (
                     <DiskVolumeCard
                        name={`${name}[${i}]`}
                        // eslint-disable-next-line react/no-array-index-key
                        key={i}
                        index={i}
                        disabled={disabled}
                        readonly={readonly}
                        onRemove={handleRemoveVolume}
                        onRestore={handleRestoreVolume}
                        layers={layers}
                        staticResources={staticResources}
                     />
                  ))
               )}

               {disabled || readonly ? null : (
                  <FormButton
                     icon={faPlus}
                     onClick={handleAddVolume}
                     skipLeftSpace={true}
                     dataE2e={'Volumes:AddVolume'}
                  >
                     Add volume
                  </FormButton>
               )}
            </div>
         </FieldLayout2>
      );
   },
);

DiskVolumesSubForm.displayName = 'DiskVolumesSubForm';
