import { YCSelect, YCSelectItem } from '@yandex-data-ui/common';
import {
   deepClone,
   EMPTY_VALUE,
   ExtendedFieldConfig,
   ExternalLink,
   FieldLayout2,
   formatDate,
   useExtendedField,
} from '@yandex-infracloud-ui/libs';
import React, { ReactNode, useCallback, useEffect, useMemo, useState } from 'react';

import { EXTERNAL_LINKS } from '../../../../../models';
import { EnvName } from '../../../../../models/environment';
import { Sidecar, ResourceState, SandboxResource } from '../../../../../models/ui';
import { sandboxApi, useConfig } from '../../../../../services';
import { SandboxResourceParams } from '../../../../../services/api/services/SandboxApi';
import { SidecarWarning } from '../../components/SidecarWarning/SidecarWarning';

import classes from './SidecarResourceField.module.css';

type SidecarResourceFieldProps = ExtendedFieldConfig<any, Sidecar['resourceRevision']>;

interface RevisionItem {
   revision: number;
   description: string | null;
   time?: SandboxResource['time'];
   releaseNotes?: string | null;
}

type RevisionsMap = Map<string, RevisionItem>;

enum ReleaseLocation {
   XDC = 'xdc',
   Acceptance = 'xdc_acceptance',
   Test = 'sas_test',
   ManPre = 'man_pre',
}

// as example https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/release_machine/components/configs/pod_agent.py?rev=7542204#L75-79
function getReleasedLocation(envName: EnvName): ReleaseLocation {
   if (envName === EnvName.ExtAcceptance) {
      return ReleaseLocation.Acceptance;
   }
   if (envName === EnvName.ExtManPre) {
      return ReleaseLocation.ManPre;
   }
   if ([EnvName.ExtTest, EnvName.ExtLocalTest].includes(envName)) {
      return ReleaseLocation.Test;
   }
   return ReleaseLocation.XDC;
}

function getResourcesParams({
   sidecarSandboxId,
   location,
}: {
   sidecarSandboxId: string;
   location?: ReleaseLocation;
}): SandboxResourceParams {
   const additionalAttributes: Record<string, string> = {};
   if (location) {
      additionalAttributes[`released_${location}`] = 'True';
   } else {
      additionalAttributes.released = 'stable';
   }
   return {
      type: sidecarSandboxId,
      order: '-time.created',
      limit: 10,
      state: ResourceState.READY,
      attributes: {
         ttl: 'inf',
         ...additionalAttributes,
      },
   };
}

export const SidecarResourceField = React.memo(
   (
      props: SidecarResourceFieldProps & { labelRevision: number | null } & { sidecarName: string } & {
         sidecarSandboxId: string;
      },
   ) => {
      const { readonly, onChange, disabled, field } = useExtendedField(props as SidecarResourceFieldProps);
      const { value } = field;
      const { labelRevision, sidecarName, sidecarSandboxId } = props;

      const [revisionItems, setRevisionItems] = useState<RevisionsMap>(new Map());

      const [currentRevisions, setCurrentRevisions] = useState<RevisionsMap>(new Map());

      useEffect(() => {
         setCurrentRevisions(previousRevisions => {
            const revisions = deepClone(previousRevisions);
            for (const [id, item] of revisionItems) {
               revisions.set(id, item);
            }

            return revisions;
         });
      }, [revisionItems]);

      const isExistValueInfo = currentRevisions.has(String(value));

      useEffect(() => {
         if (value && !isExistValueInfo) {
            sandboxApi.getResource(value).subscribe(resource => {
               const { id, description, time, attributes } = resource;
               const item: RevisionItem = {
                  revision: id,
                  description,
                  time,
                  releaseNotes: attributes?.release_notes ?? null,
               };
               setCurrentRevisions(e => {
                  const newMap = deepClone(e);
                  newMap.set(String(id), item);
                  return newMap;
               });
            });
         }
      }, [value, isExistValueInfo]);

      const items = useMemo(() => {
         const list = Array.from(currentRevisions.values());
         if (value && !currentRevisions.has(String(value))) {
            list.push({
               revision: value,
               description: null,
            });
         }
         return list.sort((a, b) => b.revision - a.revision).map(e => revisionToSelectItem(e));
      }, [currentRevisions, value]);

      const onChangeSelected = useCallback(v => onChange(Number(v)), [onChange]);

      const config = useConfig()!;
      const { envName } = config;
      const releasedLocation = useMemo(() => getReleasedLocation(envName), [envName]);

      const saveRevisionItems = useCallback((resources: SandboxResource[]) => {
         const resourceMap = new Map<string, RevisionItem>();
         for (const resource of resources) {
            const { id, description, time, attributes } = resource;
            resourceMap.set(String(id), {
               revision: id,
               description,
               time,
               releaseNotes: attributes?.release_notes ?? null,
            });
         }
         setRevisionItems(resourceMap);
      }, []);

      const loadRevisions = useCallback(() => {
         sandboxApi
            .getResources(getResourcesParams({ sidecarSandboxId, location: releasedLocation }))
            .subscribe(resources => saveRevisionItems(resources));
      }, [releasedLocation, saveRevisionItems, sidecarSandboxId]);

      const setLabelRevision = useCallback(() => {
         onChange(labelRevision);
      }, [labelRevision, onChange]);

      const releaseNotes = useMemo(() => {
         if (!value) {
            return null;
         }
         if (!currentRevisions.has(String(value))) {
            return null;
         }
         const { releaseNotes: currentReleaseNotes } = currentRevisions.get(String(value))!;
         if (!currentReleaseNotes) {
            return null;
         }
         const links: string[] = [];
         const separator = String(Math.random()).slice(2);
         const text = currentReleaseNotes
            .replace(/https:\/\/.*yandex-team\.ru(\/.*)?/g, link => {
               links.push(link);
               return separator;
            })
            .split(separator);
         const elements: ReactNode[] = [text[0]];
         for (let i = 1; i < text.length; i += 1) {
            elements.push(<ExternalLink href={links[i - 1]}>{links[i - 1]}</ExternalLink>, text[i]);
         }
         return elements;
      }, [value, currentRevisions]);

      return (
         <FieldLayout2 {...props} label={'Version'} readonlyDots={readonly}>
            {readonly ? (
               <div className={classes.readonly}>{value ?? EMPTY_VALUE}</div>
            ) : (
               <>
                  <div onClick={loadRevisions} role={'button'} tabIndex={0} aria-label={'Select revision'}>
                     <YCSelect
                        type={'single'}
                        onUpdate={onChangeSelected}
                        value={String(value ?? '')}
                        items={items}
                        showSearch={false}
                        placeholder={'Select revision'}
                        className={classes.control}
                        disabled={disabled}
                        showItemMeta={true}
                     />
                  </div>

                  {releaseNotes && (
                     <>
                        <br />
                        Release notes:
                        <p>{releaseNotes}</p>
                     </>
                  )}

                  {labelRevision && labelRevision !== value && (
                     <SidecarWarning
                        currentRevision={value}
                        labelRevision={labelRevision!}
                        setLabelRevision={setLabelRevision}
                        sidecarName={sidecarName}
                     />
                  )}
               </>
            )}
         </FieldLayout2>
      );
   },
);

SidecarResourceField.displayName = 'SidecarResourceField';

function revisionToSelectItem(revisionItem: RevisionItem): YCSelectItem {
   return {
      key: String(revisionItem.revision),
      title: `${
         revisionItem.time?.created ? formatDate(new Date(revisionItem.time.created), 'dd.MM.yyyy HH:mm') : ''
      } (${revisionItem.revision})`,
      url: EXTERNAL_LINKS.sandboxResource(revisionItem.revision),
      value: String(revisionItem.revision),
      meta: revisionItem.description ?? '',
   };
}
