import { YCSelect, YCSelectItem } from '@yandex-data-ui/common';
import { EMPTY_VALUE, FieldLayout2, useDismounted } from '@yandex-infracloud-ui/libs';
import { getIn, useFormikContext } from 'formik';
import React, { useCallback, useEffect, useMemo, useState } from 'react';
import { takeUntil } from 'rxjs/operators';

import { Nullable } from '../../../../../models/typeHelpers';
import { InfraRecord } from '../../../../../models/ui';
import { infraApi } from '../../../../../services';

import classes from './InfraEnvironmentSubForm.module.css';

interface Props {
   name: string;

   readonly: boolean;
   disabled: boolean;
   service: number | null;

   onChange(v: Nullable<InfraRecord>): void;
}

export const InfraEnvironmentSubForm: React.FC<Props> = React.memo(props => {
   const { name, readonly, disabled, onChange, service } = props;
   const form = useFormikContext();
   const value: InfraRecord = getIn(form.values, name);

   const dismounted = useDismounted();

   const [items, setItems] = useState<YCSelectItem[]>([]);

   const environmentName = useMemo(() => {
      if (value.name) {
         return value.name;
      }

      const { id } = value;
      if (!id) {
         return '';
      }

      const title = items.find(i => id.toString() === i.value)?.title;

      return typeof title === 'string' ? title : '';
   }, [items, value]);

   useEffect(() => {
      // идём в ручку саджеста только если:
      // 1) не readonly (для nanny)
      // 2) если есть id, но нет name (для deploy)
      if (!readonly || (value.id && !value.name)) {
         infraApi
            .getEnvironments(service ?? undefined)
            .pipe(takeUntil(dismounted))
            .subscribe(x => {
               setItems(
                  x?.map(
                     v =>
                        ({
                           key: v.id,
                           value: v.id,
                           meta: `id: ${v.id}`,
                           title: v.name,
                        } as YCSelectItem),
                  ) ?? [],
               );
            });
      }
   }, [dismounted, setItems, service, readonly, value.id, value.name]);

   useEffect(() => {
      if (!readonly && value.id && environmentName !== value.name) {
         // добавим name в spec, если его там нет (для deploy)
         onChange({
            id: value.id,
            name: environmentName,
         });
      }
   }, [readonly, value.id, value.name, environmentName, onChange]);

   const handleOnChange = useCallback(
      (v: string | null, { items: item }: { items: YCSelectItem; isOutsideClick: boolean }) => {
         if (!item?.value) {
            onChange({
               id: null,
               name: null,
            });
            return;
         }

         onChange({
            id: Number(item.value) ?? null,
            name: typeof item.title === 'string' ? item.title : null,
         });
      },
      [onChange],
   );

   return (
      <div data-e2e={'Infra:Environment'}>
         <FieldLayout2 name={name} label={'Environment'} readonlyDots={readonly} readonly={readonly}>
            {readonly ? (
               <div className={classes.readonly}>{value.id ? `${environmentName} (id: ${value.id})` : EMPTY_VALUE}</div>
            ) : (
               <>
                  <YCSelect
                     disabled={disabled || readonly || !service}
                     items={items}
                     onUpdate={handleOnChange}
                     value={value.id?.toString() ?? undefined}
                     placeholder={'Select environment'}
                     type={'single'}
                     showItemMeta={true}
                     allowEmptyValue={true}
                  />

                  {value.id && <div className={classes.details}>id: {value.id}</div>}
               </>
            )}
         </FieldLayout2>
      </div>
   );
});

InfraEnvironmentSubForm.displayName = 'InfraEnvironmentSubForm';
