import { ExternalLink, FieldLayout2, WarningPlate } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback } from 'react';

import { EXTERNAL_LINKS } from '../../../../../models';
import { Infra, InfraRecord } from '../../../../../models/ui';

import { InfraEnvironmentSubForm } from './InfraEnvironmentSubForm';
import { InfraServiceSubForm } from './InfraServiceSubForm';
import classes from './InfraSubForm.module.css';

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;
   isApprovalRequired: boolean;
}

export const InfraSubForm: React.FC<Props> = React.memo(({ readonly, disabled, name, isApprovalRequired }) => {
   const form = useFormikContext();
   const { value } = form.getFieldMeta<Infra>(name);

   const approvalRequiredOrDisabled = isApprovalRequired || disabled;

   const onServiceChange = useCallback(
      (v: InfraRecord) => {
         const newValue =
            v.id === value.service.id
               ? {
                    ...value,
                    service: {
                       id: v.id,
                       name: v.name,
                    },
                 }
               : {
                    service: {
                       id: v.id,
                       name: v.name,
                    },
                    environment: {
                       id: null,
                       name: null,
                    },
                 };

         form.setFieldValue(name, newValue);
      },
      [form, name, value],
   );

   const onEnvironmentChange = useCallback(
      (v: InfraRecord) => {
         const newValue = {
            ...value,
            environment: {
               id: v.id,
               name: v.name,
            },
         };

         form.setFieldValue(name, newValue);
      },
      [form, name, value],
   );

   return (
      <FieldLayout2
         name={name}
         label={'Infra notifications'}
         readonly={readonly}
         bigLabel={true}
         hint={
            <>
               <p>
                  <ExternalLink href={EXTERNAL_LINKS.infra()}>infra.yandex-team.ru</ExternalLink>
               </p>

               <p>
                  See <ExternalLink href={EXTERNAL_LINKS.wiki.infra}>Yandex Infra FAQ</ExternalLink> for details.
               </p>
            </>
         }
      >
         <div className={classes.wrapper} data-e2e={'Infra'}>
            {isApprovalRequired && !readonly && (
               <WarningPlate className={classes.warningPlate}>
                  Approval policy is applied. Currently and unfortunately Deploy UI is unable to edit labels for drafts.
                  You can use <ExternalLink href={EXTERNAL_LINKS.deployDocs.dctl}>DCTL</ExternalLink> to edit the Infra
                  notifications settings.
               </WarningPlate>
            )}
            <InfraServiceSubForm
               name={`${name}.service`}
               readonly={readonly}
               disabled={approvalRequiredOrDisabled}
               onChange={onServiceChange}
            />

            <InfraEnvironmentSubForm
               key={value.service.id ?? ''}
               name={`${name}.environment`}
               readonly={readonly}
               disabled={approvalRequiredOrDisabled}
               onChange={onEnvironmentChange}
               service={value.service.id}
            />
         </div>
      </FieldLayout2>
   );
});

InfraSubForm.displayName = 'InfraSubForm';
