import {
   YCSelect,
   YCSelectGetItemsArgs,
   YCSelectGetItemsArgsFetchItems,
   YCSelectGetItemsArgsInitItems,
   YCSelectItem,
} from '@yandex-data-ui/common';
import {
   EMPTY_VALUE,
   ExternalLink,
   FieldLayout2,
   InputField2,
   isEmpty,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback, useMemo } from 'react';
import { takeUntil } from 'rxjs/operators';

import { EXTERNAL_LINKS, urlBuilder } from '../../../../../models';
import { ProjectItem } from '../../../../../models/ui';
import { useAbcService } from '../../../../../redux/hooks';
import { ypApi } from '../../../../../services';
import { logPromiseError } from '../../../../../utils';
import { ProjectIdHint } from '../../hints';

import classes from './ProjectSubForm.module.css';

interface ProjectSelectItem extends YCSelectItem {
   project: ProjectItem;
}

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;
}

export const ProjectSubForm: React.FC<Props> = React.memo(({ readonly, disabled, name }) => {
   const form = useFormikContext();
   const { value } = form.getFieldMeta<ProjectItem | null>(name);

   const abcId = useMemo(() => value?.accountId?.split(':')?.[2], [value?.accountId]);

   const abcService = useAbcService(abcId ?? null);

   const dismounted = useDismounted();

   const getItems = useCallback(
      (args: YCSelectGetItemsArgs) => {
         const initArgs = args as YCSelectGetItemsArgsInitItems;
         if (!isEmpty(initArgs.exactKeys)) {
            return Promise.resolve({
               items: initArgs.exactKeys.map(v =>
                  value?.id === v ? projectToSelectItem(value) : ({ key: v, value: v, title: v } as YCSelectItem),
               ),
            });
         }

         const fetchArgs = args as YCSelectGetItemsArgsFetchItems;
         return (
            ypApi
               .selectProjects({
                  substring: fetchArgs.searchPattern ?? '',
                  paths: e => [e.meta.id, e.spec.account_id],
               })
               .pipe(takeUntil(dismounted))
               .toPromise()
               // toPromise for canceled observable emits `undefined` value.
               // So, I have to handle it to avoid returning undefined.
               .then(x => x?.values ?? [])
               .then(projects =>
                  projects.map(e =>
                     projectToSelectItem({
                        id: e.meta!.id!,
                        accountId: e.spec!.account_id!,
                     }),
                  ),
               )
               .then(items => ({ items }), logPromiseError)
         );
      },
      [dismounted, value],
   );

   const handleOnChange = useCallback(
      (rawValue: string | null, { items: item }: { items: YCSelectItem; isOutsideClick: boolean }) => {
         const newProject = (item as ProjectSelectItem).project;

         form.setFieldValue(name, {
            id: newProject?.id,
            accountId: newProject?.accountId,
         });
      },
      [form, name],
   );

   return (
      <>
         <FieldLayout2
            name={name}
            label={'Project ID'}
            hint={<ProjectIdHint />}
            disabled={disabled}
            readonly={readonly}
            readonlyDots={readonly}
            required={true}
         >
            {readonly ? (
               <div className={classes.readonlyValue}>{value?.id || EMPTY_VALUE}</div>
            ) : (
               <YCSelect
                  className={classes.control}
                  disabled={disabled}
                  getItems={getItems}
                  onUpdate={handleOnChange}
                  placeholder={'Select project...'}
                  type={'single'}
                  value={value?.id}
               />
            )}
         </FieldLayout2>

         <InputField2
            name={`${name}.accountId`}
            label={'Account ID'}
            disabled={true}
            readonly={readonly}
            readonlyDots={readonly}
            help={
               abcService && (
                  <ExternalLink href={EXTERNAL_LINKS.abcService(abcService.slug)}>{abcService.name}</ExternalLink>
               )
            }
         />
      </>
   );
});

ProjectSubForm.displayName = 'ProjectSubForm';

function projectToSelectItem(p: ProjectItem): ProjectSelectItem {
   return {
      key: p.id,
      project: p,
      title: p.id,
      url: urlBuilder.project(p.id),
      value: p.id,
   };
}
