import { faUndo } from '@fortawesome/free-solid-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { CheckboxField2, classNames, FieldLayout2, FormCard, InputField2 } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React from 'react';

import { LegoButton } from '../../../../../_lego';
import { workloadDefaultValues, WorkloadUnistatYasmParams } from '../../../../../models/ui';
import { getFilledValue } from '../../../../../utils';
import { YasmTagsSubForm } from '../../../_common/YasmTagsSubForm/YasmTagsSubForm';
import { YasmTagsInheritanceHint } from '../../hints';

import classes from './YasmUnistatRecord.module.css';

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;

   onRemove(): void;

   onRestore(): void;
}

// const outputOptions: EnumOption[] = [
//    TMonitoringUnistatEndpoint_EOutputFormat.OF_YASM_JSON,
//    TMonitoringUnistatEndpoint_EOutputFormat.OF_YASM_PROTO,
//    TMonitoringUnistatEndpoint_EOutputFormat.OF_SOLOMON,
// ].map(e => ({
//    value: e,
//    title: e,
// }));

const defaultValues = workloadDefaultValues.yasm.unistat;

export const YasmUnistatRecord: React.FC<Props> = React.memo(({ readonly, disabled, name, onRemove, onRestore }) => {
   const form = useFormikContext();
   const { value } = form.getFieldMeta<WorkloadUnistatYasmParams>(name);
   const {
      url,
      port,
      form: { removed, old },
   } = value;

   const title = (
      <div className={classNames(classes.header, { [classes.removed]: removed })}>
         <span className={classes.title}>
            {port ? `:${port}` : null}
            {getFilledValue(url, '', defaultValues.path)}
         </span>
         {!old && <span className={classes.newLabel}>new</span>}
      </div>
   );

   const record = (
      <div className={classes.record}>
         <FieldLayout2 name={name} label={null} readonly={readonly} bigLabel={true}>
            <div className={classNames({ [classes.unistatRow]: !readonly, [classes.readonly]: readonly })}>
               <InputField2
                  name={`${name}.url`}
                  label={'Unistat path'}
                  controlProps={{ placeholder: defaultValues.path }}
                  readonly={readonly}
                  disabled={disabled}
                  hideErrors={true}
                  readonlyDots={readonly}
               />

               <InputField2
                  name={`${name}.port`}
                  label={'Port'}
                  controlProps={{
                     controlProps: { min: 0 },
                     type: 'number',
                     className: classes.port,
                  }}
                  readonly={readonly}
                  disabled={disabled}
                  hideErrors={true}
                  readonlyDots={readonly}
               />
               <InputField2
                  name={`${name}.prefix`}
                  label={'Prefix'}
                  controlProps={{ className: classes.prefix, placeholder: defaultValues.prefix }}
                  readonly={readonly}
                  disabled={disabled}
                  hideErrors={true}
                  readonlyDots={readonly}
               />
            </div>

            {/*  Скрываем до требования */}
            {/* <EnumField2
               name={`${name}.outputFormat`}
               label={'Output format'}
               controlProps={{ cls: classes.output, options: outputOptions } as any}
               readonly={readonly}
               disabled={disabled}
               hideErrors={true}
               readonlyDots={readonly}
            /> */}

            <YasmTagsSubForm name={`${name}.yasmTags`} readonly={readonly} disabled={disabled} />

            <CheckboxField2
               name={`${name}.inheritMissedLabels`}
               disabled={disabled}
               readonly={readonly}
               readonlyDots={readonly}
               label={'Labels inheritance'}
               controlProps={{ disableIndeterminate: true }}
               hint={<YasmTagsInheritanceHint />}
            >
               Inherit missing labels
            </CheckboxField2>
         </FieldLayout2>
      </div>
   );

   return (
      <FormCard title={title} hasRemoveButton={!readonly && !removed} onRemove={onRemove}>
         {removed ? (
            <div className={classNames(classes.record, classes.removed)}>
               <span>(removed)</span>

               <LegoButton theme={'clear'} title={'Restore'} onClick={onRestore}>
                  <FontAwesomeIcon icon={faUndo} />
               </LegoButton>
            </div>
         ) : (
            record
         )}
      </FormCard>
   );
});

YasmUnistatRecord.displayName = 'YasmUnistatRecord';
