import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button } from '@yandex-cloud/uikit';
import { EMPTY_VALUE, FieldLayout2, isEmpty } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback, useContext } from 'react';

import { entityLevel, getEmptyWorkloadUnistat, WorkloadUnistatYasmParams } from '../../../../../models/ui';
import { HugeFormContext } from '../../../../huge-form';
import { YasmUnistatHint } from '../../hints';

import { YasmUnistatRecord } from './YasmUnistatRecord';

import classes from './YasmUnistatSubForm.module.css';

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;
}

export const YasmUnistatSubForm = React.memo(({ disabled, name, readonly }: Props) => {
   const context = useContext(HugeFormContext);
   const form = useFormikContext();
   const { value } = form.getFieldMeta<WorkloadUnistatYasmParams[]>(name);

   const handleAddUnistat = useCallback(() => {
      const workloadEmptyParams = context.currentEmptyParams.get(entityLevel.workload);
      const newValue: WorkloadUnistatYasmParams[] = [
         ...value,
         {
            ...getEmptyWorkloadUnistat({ projectId: workloadEmptyParams?.projectId as string | undefined }),
            form: { old: false, removed: false, index: value.length },
         },
      ];

      const last = newValue.length - 1;

      form.setFieldValue(name, newValue);
      form.setFieldTouched(`${name}[${last}].id`, true);
   }, [form, name, value, context.currentEmptyParams]);

   const handleRemoveUnistat = useCallback(
      (i: number) => {
         if (value[i].form.old) {
            form.setFieldValue(`${name}[${i}].form.removed`, true);
         } else {
            const newValue = [...value];

            newValue.splice(i, 1); // TODO: @nikolaichev сделать нормализацию на уровне формы

            form.setFieldTouched(`${name}[${i}]`, false);
            form.setFieldValue(
               name,
               newValue.map(e => {
                  if (e.form.index >= i) {
                     e.form.index -= 1;
                  }
                  return e;
               }),
            );
         }
      },
      [form, name, value],
   );

   const handleRestoreUnistat = useCallback(
      (i: number) => {
         form.setFieldValue(`${name}[${i}].form.removed`, false);
      },
      [form, name],
   );

   return (
      <FieldLayout2
         name={name}
         label={'Unistats'}
         hint={<YasmUnistatHint />}
         readonly={readonly}
         bigLabel={true}
         hideErrors={true}
      >
         <div className={classes.unistat}>
            {!isEmpty(value) ? (
               <div className={classes.list}>
                  {value.map((unistat, i) => (
                     // eslint-disable-next-line react/no-array-index-key
                     <div key={i} className={classes.record}>
                        <YasmUnistatRecord
                           name={`${name}[${i}]`}
                           disabled={disabled}
                           readonly={readonly}
                           onRemove={() => handleRemoveUnistat(i)}
                           onRestore={() => handleRestoreUnistat(i)}
                        />
                     </div>
                  ))}
               </div>
            ) : readonly ? (
               EMPTY_VALUE
            ) : null}

            {disabled || readonly ? null : (
               <div className={classes.add}>
                  <Button onClick={handleAddUnistat} qa={'Unistat:AddUnistatButton'}>
                     <FontAwesomeIcon icon={faPlus} /> Add unistat
                  </Button>
               </div>
            )}
         </div>
      </FieldLayout2>
   );
   /* eslint-enable no-underscore-dangle */
});

YasmUnistatSubForm.displayName = 'YasmUnistatSubForm';
