import { classNames, FieldLayout2, FormAdvancedSection, InputField2 } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback, useState } from 'react';

import { SidecarName, Workload, WorkloadFormParams } from '../../../../../models/ui';
import { getInitialIds } from '../../../../../models/ui/stage/form-utils';
import { EnabledSwitcher, EnabledSwitcherField } from '../../../../forms';
import { SubForm } from '../../../../huge-form';
import { SidecarResourcePlate } from '../../../../stage-huge-form/components';

import { CoredumpHint } from '../../hints';

import classes from './CoredumpSubForm.module.css';

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;
   form?: SubForm<WorkloadFormParams, Workload, any>;
}

export const CoredumpSubForm: React.FC<Props> = React.memo(({ readonly, disabled, name, form: subForm }) => {
   const [opened, setOpened] = useState(false);
   const form = useFormikContext<WorkloadFormParams>();

   const { deployUnit = '', box = '', workload = '' } = subForm
      ? getInitialIds(subForm.formParams, subForm.parentForms, subForm.levelConfig.id)
      : {};

   const value = form.values.coredumpPolicy;

   const localDisabled = disabled || !value.enabled;

   const handleEnabledChange = useCallback(
      (v: boolean) => {
         form.setFieldValue(`${name}.enabled`, v);
         setOpened(v);
      },
      [form, name],
   );

   return (
      <FieldLayout2
         name={'coredumpPolicy'}
         label={'Coredump Processing'}
         hint={<CoredumpHint />}
         hideErrors={true}
         readonly={readonly}
         bigLabel={true}
      >
         <FormAdvancedSection
            header={
               !readonly || !value.enabled ? (
                  <EnabledSwitcher
                     value={value.enabled}
                     name={`${name}.enabled`}
                     onChange={handleEnabledChange}
                     readonly={readonly}
                     disabled={disabled}
                  />
               ) : null
            }
            hasValue={false}
            openedByDefault={opened || value.enabled}
            hideToggleSwitch={!value.enabled || readonly}
         >
            {value.enabled && (
               <SidecarResourcePlate
                  sidecar={SidecarName.Coredump}
                  deployUnit={deployUnit}
                  box={box}
                  workload={workload}
               />
            )}
            <div className={classes.form}>
               <FieldLayout2
                  name={name}
                  label={'Coredump Policy'}
                  readonly={readonly}
                  bigLabel={true}
                  hideErrors={true}
               >
                  <div className={classNames({ [classes.limitsRow]: !readonly })}>
                     <InputField2
                        name={`${name}.countLimit`}
                        label={'Count limit'}
                        controlProps={{ controlProps: { min: 0 }, type: 'number' } as any}
                        required={true}
                        disabled={localDisabled}
                        readonly={readonly}
                        readonlyDots={readonly}
                     />

                     <InputField2
                        name={`${name}.totalSizeLimitMegabytes`}
                        label={'Total size limit megabytes'}
                        controlProps={{ controlProps: { min: 0 }, type: 'number' } as any}
                        required={true}
                        disabled={localDisabled}
                        readonly={readonly}
                        readonlyDots={readonly}
                     />

                     <InputField2
                        name={`${name}.probability`}
                        label={'Probability(%)'}
                        controlProps={{ controlProps: { min: 0 }, type: 'number' } as any}
                        disabled={localDisabled}
                        readonly={readonly}
                        readonlyDots={readonly}
                     />

                     <InputField2
                        name={`${name}.cleanupTtlSeconds`}
                        label={'Cleanup ttl seconds'}
                        controlProps={{ controlProps: { min: 0 }, type: 'number' } as any}
                        disabled={localDisabled}
                        readonly={readonly}
                        readonlyDots={readonly}
                     />
                  </div>
               </FieldLayout2>

               <FieldLayout2 name={`${name}.aggregator`} label={'Aggregation'} readonly={readonly} bigLabel={true}>
                  {!readonly || !value?.aggregator?.enabled ? (
                     <EnabledSwitcherField
                        name={`${name}.aggregator.enabled`}
                        label={null}
                        readonly={readonly}
                        disabled={disabled}
                     />
                  ) : null}

                  {value.aggregator?.enabled && (
                     <>
                        <InputField2
                           name={`${name}.aggregator.url`}
                           label={'URL to aggregator'}
                           controlProps={{
                              placeholder: 'https://coredumps.n.yandex-team.ru/submit_core',
                              className: classes.aggregatorURL,
                           }}
                           disabled={localDisabled}
                           readonly={readonly}
                           readonlyDots={readonly}
                        />

                        <div className={classNames({ [classes.aggregatorRow]: !readonly })}>
                           <InputField2
                              name={`${name}.aggregator.serviceName`}
                              label={'Service name tag for trace'}
                              controlProps={{
                                 placeholder: 'deploy.stageId.workloadId',
                                 className: classes.aggregatorField,
                              }}
                              disabled={localDisabled}
                              readonly={readonly}
                              readonlyDots={readonly}
                           />

                           <InputField2
                              name={`${name}.aggregator.ctype`}
                              label={'Ctype tag for trace'}
                              controlProps={{
                                 placeholder: 'empty by default',
                                 className: classes.aggregatorField,
                              }}
                              disabled={localDisabled}
                              readonly={readonly}
                              readonlyDots={readonly}
                           />
                        </div>
                     </>
                  )}
               </FieldLayout2>

               {/* <InputField2
               name={`${name}.output`}
               label={'Custom path for storing coredumps'}
               controlProps={{
                  placeholder: '/coredumps_box/<stageId>_<workloadId>-<timestamp_ms> by default'
               }}
               readonly={readonly}
               disabled={localDisabled}
            /> */}
            </div>
         </FormAdvancedSection>
      </FieldLayout2>
   );
});

CoredumpSubForm.displayName = 'CoredumpSubForm';
