import { faFileCode as faFileCodeFill } from '@fortawesome/free-solid-svg-icons';
import { faFileCode as faFileCodeEmpty, faPlus, faTrashAlt } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button, TabsDirection } from '@yandex-cloud/uikit';
import { FieldLayout2 } from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React, { useCallback } from 'react';

import { isDestroyEmpty, isExecEmpty, isProbeEmpty, isStopEmpty, WorkloadCommands } from '../../../../../models/ui';
import { nestedObjectValues } from '../../../../../utils/helpers/nestedObjectValues';
import { FormTabs } from '../../../../forms';

import { CommandFormTab } from './CommandFormTab';
import { AdvancedFieldSet } from './subForms/AdvancedSettingsSubForm';
import { CommandSubForm } from './subForms/CommandSubForm';
import { DestroySubForm } from './subForms/DestroySubForm';
import { ProbeSubForm } from './subForms/ProbeSubForm';
import { StopSubForm } from './subForms/StopSubForm';

import classes from './WorkloadCommandsSubForm.module.css';

const icons = {
   emptyIcon: faFileCodeEmpty,
   fillIcon: faFileCodeFill,
};

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;
}

export const WorkloadCommandsSubForm: React.FC<Props> = React.memo(({ readonly, disabled, name }) => {
   const form = useFormikContext();
   const { value } = form.getFieldMeta<WorkloadCommands>(name);
   const error = form.getFieldMeta<WorkloadCommands>(name).error as Partial<WorkloadCommands>;

   const handleAddInitCommand = useCallback(() => {
      form.setFieldValue(`${name}.init`, [...value.init, { command: '' }]);
   }, [form, name, value.init]);

   const handleRemoveInitCommand = useCallback(
      (i: number) => (event: React.MouseEvent<HTMLButtonElement, MouseEvent>) => {
         event.stopPropagation();
         const init = [...value.init];
         init.splice(i, 1);

         form.setFieldTouched(`${name}.init[${i}]`, false);
         form.setFieldValue(`${name}.init`, init);
      },
      [form, name, value.init],
   );

   const initCards = value.init.map((command, i, inits) => (
      <CommandFormTab
         // eslint-disable-next-line react/no-array-index-key
         key={`init_${i}`}
         id={`init_${i}`}
         title={inits.length > 1 ? `Init command ${i + 1}` : 'Init command'}
         errorsCount={nestedObjectValues(error?.init && error.init[i]).length}
         isEmpty={isExecEmpty(command)}
         dataTest={`${name}.init_${i}.tab`}
         {...icons}
      >
         <FieldLayout2
            label={
               <div className={classes.label}>
                  {inits.length > 1 ? (
                     <>
                        {`Init command ${i + 1}`}{' '}
                        <div data-test={`${name}.init_${i}.remove`}>
                           <button className={classes.iconButton} onClick={handleRemoveInitCommand(i)} type={'button'}>
                              <FontAwesomeIcon icon={faTrashAlt} />
                           </button>
                        </div>
                     </>
                  ) : (
                     'Init command'
                  )}
               </div>
            }
            name={`${name}.init[${i}]`}
         >
            <CommandSubForm
               name={`${name}.init[${i}]`}
               dataE2e={`commands.init[${i}]`}
               readonly={readonly}
               disabled={disabled}
            />
         </FieldLayout2>
      </CommandFormTab>
   ));

   const startCard = (
      <CommandFormTab
         id={'start'}
         title={'Start command'}
         errorsCount={nestedObjectValues(error?.start).length}
         isEmpty={isExecEmpty(value.start)}
         dataTest={`${name}.start.tab`}
         {...icons}
      >
         <FieldLayout2 label={<div className={classes.label}>Start command</div>} name={`${name}.start`}>
            <CommandSubForm
               name={`${name}.start`}
               dataE2e={`${name}.start`}
               readonly={readonly}
               disabled={disabled}
               sections={new Set([AdvancedFieldSet.Access, AdvancedFieldSet.Limits])}
            />
         </FieldLayout2>
      </CommandFormTab>
   );

   const livenessCard = (
      <CommandFormTab
         id={'liveness'}
         title={'Liveness probe'}
         errorsCount={nestedObjectValues(error?.liveness).length}
         isEmpty={isProbeEmpty(value.liveness)}
         dataTest={`${name}.liveness.tab`}
         {...icons}
      >
         <FieldLayout2 label={<div className={classes.label}>Liveness probe</div>} name={`${name}.liveness`}>
            <ProbeSubForm
               name={`${name}.liveness`}
               dataE2e={'commands.liveness'}
               readonly={readonly}
               disabled={disabled}
            />
         </FieldLayout2>
      </CommandFormTab>
   );

   const readinessCard = (
      <CommandFormTab
         id={'readiness'}
         title={'Readiness probe'}
         errorsCount={nestedObjectValues(error?.readiness).length}
         isEmpty={isProbeEmpty(value.readiness)}
         dataTest={`${name}.readiness.tab`}
         {...icons}
      >
         <FieldLayout2 label={<div className={classes.label}>Readiness probe</div>} name={`${name}.readiness`}>
            <ProbeSubForm
               name={`${name}.readiness`}
               dataE2e={'commands.readiness'}
               readonly={readonly}
               disabled={disabled}
            />{' '}
         </FieldLayout2>
      </CommandFormTab>
   );

   const stopCard = (
      <CommandFormTab
         id={'stop'}
         title={'Stop policy'}
         errorsCount={nestedObjectValues(error?.stop).length}
         isEmpty={isStopEmpty(value.stop)}
         dataTest={`${name}.stop.tab`}
         {...icons}
      >
         <FieldLayout2 label={<div className={classes.label}>Stop policy</div>} name={`${name}.stop`}>
            <StopSubForm name={`${name}.stop`} dataE2e={'commands.stop'} readonly={readonly} disabled={disabled} />
         </FieldLayout2>
      </CommandFormTab>
   );

   const destroyCard = (
      <CommandFormTab
         id={'destroy'}
         title={'Destroy policy'}
         errorsCount={nestedObjectValues(error?.destroy).length}
         isEmpty={isDestroyEmpty(value.destroy)}
         dataTest={`${name}.destroy.tab`}
         {...icons}
      >
         <FieldLayout2 label={<div className={classes.label}>Destroy policy</div>} name={`${name}.destroy`}>
            <DestroySubForm
               name={`${name}.destroy`}
               dataE2e={'commands.destroy'}
               readonly={readonly}
               disabled={disabled}
            />
         </FieldLayout2>
      </CommandFormTab>
   );

   const actions = !readonly && (
      <>
         <div className={classes.spacer} />
         <div className={classes.button}>
            <Button qa={'commands.init.buttons.add'} disabled={disabled} onClick={handleAddInitCommand}>
               <FontAwesomeIcon icon={faPlus} /> Add init command
            </Button>
         </div>
      </>
   );

   return (
      <div className={classes.wrapper}>
         <FormTabs direction={TabsDirection.Vertical} actions={actions}>
            {initCards}
            {startCard}
            {livenessCard}
            {readinessCard}
            {stopCard}
            {destroyCard}
         </FormTabs>
      </div>
   );
});

WorkloadCommandsSubForm.displayName = 'WorkloadCommandsSubForm';
