import {
   BytesField,
   DurationField,
   DurationFieldControlProps,
   ExternalLink,
   FormAdvancedSection,
   Hint,
   InputField2,
} from '@yandex-infracloud-ui/libs';
import { useFormikContext } from 'formik';
import React from 'react';

import { EXTERNAL_LINKS } from '../../../../../../models';
import { isWorkloadAdvancedSettingsEmpty, WorkloadCommandWithAdvancedSettings } from '../../../../../../models/ui';
import { CpuLimitField, FormFieldSet } from '../../../../../forms';

import classes from './AdvancedSettingsSubForm.module.css';
import { renderBytesHelp, renderCpuHelp, renderDurationHelp } from './fieldHelp';

function RetryWikiLink() {
   return (
      <Hint
         text={
            <>
               See <ExternalLink href={EXTERNAL_LINKS.deployDocs.probeTimeLimit}>Docs</ExternalLink>
            </>
         }
      />
   );
}

const durationControlProps: DurationFieldControlProps = {
   useRenderDetails: false,
   useMilliseconds: true,
};

export enum RetryPolicyFields {
   InitialDelayMs,
   MinRestartPeriodMs,
   MaxRestartPeriodMs,
   MaxExecutionTimeMs,
   RestartPeriodBackOff,
   RestartPeriodScaleMs,
}

interface SubFormProps {
   disabled: boolean;
   name: string;
   readonly: boolean;
   disabledFields?: RetryPolicyFields[];
}

function RetryPolicySubForm({ name, disabled, readonly, disabledFields = [] }: SubFormProps) {
   return (
      <FormFieldSet
         title={
            <>
               Retry policy <RetryWikiLink />
            </>
         }
         bodyClassName={classes.advancedRetryPolicy}
      >
         <DurationField
            name={`${name}.retryPolicy.initialDelayMs`}
            label={'Initial delay'}
            hint={<>5 seconds if empty or zero (for readiness/liveness)</>}
            help={renderDurationHelp}
            controlProps={durationControlProps}
            readonly={readonly}
            disabled={disabled || disabledFields.includes(RetryPolicyFields.InitialDelayMs)}
            hideErrors={true}
         />

         <DurationField
            name={`${name}.retryPolicy.minRestartPeriodMs`}
            label={'Min restart period'}
            hint={<>30 seconds if empty or zero</>}
            help={renderDurationHelp}
            controlProps={durationControlProps}
            readonly={readonly}
            disabled={disabled || disabledFields.includes(RetryPolicyFields.MinRestartPeriodMs)}
            hideErrors={true}
         />

         <DurationField
            name={`${name}.retryPolicy.maxRestartPeriodMs`}
            label={'Max restart period'}
            hint={
               <>
                  <code>UINT64_MAX</code> if empty or zero
               </>
            }
            help={renderDurationHelp}
            controlProps={durationControlProps}
            readonly={readonly}
            disabled={disabled || disabledFields.includes(RetryPolicyFields.MaxRestartPeriodMs)}
            hideErrors={true}
         />

         <DurationField
            name={`${name}.retryPolicy.maxExecutionTimeMs`}
            label={'Max execution'}
            hint={<>30 minutes if empty or zero</>}
            help={renderDurationHelp}
            controlProps={durationControlProps}
            readonly={readonly}
            disabled={disabled || disabledFields.includes(RetryPolicyFields.MaxExecutionTimeMs)}
            hideErrors={true}
         />

         <InputField2
            name={`${name}.retryPolicy.restartPeriodBackOff`}
            label={'Restart period backoff'}
            controlProps={{ type: 'number', controlProps: { min: 0 } }}
            readonly={readonly}
            disabled={disabled || disabledFields.includes(RetryPolicyFields.RestartPeriodBackOff)}
            hideErrors={true}
         />

         <DurationField
            name={`${name}.retryPolicy.restartPeriodScaleMs`}
            label={'Restart period scale'}
            help={renderDurationHelp}
            controlProps={durationControlProps}
            readonly={readonly}
            disabled={disabled || disabledFields.includes(RetryPolicyFields.RestartPeriodScaleMs)}
            hideErrors={true}
         />
      </FormFieldSet>
   );
}

function LimitsSubForm({ name, disabled, readonly }: SubFormProps) {
   return (
      <FormFieldSet title={'Limits'} bodyClassName={classes.advancedLimits}>
         <div data-test={'AdvancedSettings:Limits:Ram'}>
            <BytesField
               name={`${name}.limits.ramLimit`}
               label={'RAM limit'}
               controlProps={{ useRenderDetails: false }}
               help={renderBytesHelp}
               readonly={readonly}
               disabled={disabled}
               hideErrors={true}
            />
         </div>

         <div data-test={'AdvancedSettings:Limits:Cpu'}>
            <CpuLimitField
               name={`${name}.limits.cpuLimit`}
               label={'CPU limit'}
               controlProps={{ useRenderDetails: false }}
               help={renderCpuHelp}
               readonly={readonly}
               disabled={disabled}
               hideErrors={true}
            />
         </div>
      </FormFieldSet>
   );
}

function AccessSubForm({ name, disabled, readonly }: SubFormProps) {
   return (
      <FormFieldSet title={'Run access'} bodyClassName={classes.advancedAccess}>
         <div data-test={'AdvancedSettings:Access:User'}>
            <InputField2
               name={`${name}.access.user`}
               label={'User'}
               readonly={readonly}
               disabled={disabled}
               hideErrors={true}
            />
         </div>

         <div data-test={'AdvancedSettings:Access:Group'}>
            <InputField2
               name={`${name}.access.group`}
               label={'Group'}
               readonly={readonly}
               disabled={disabled}
               hideErrors={true}
            />
         </div>
      </FormFieldSet>
   );
}

export enum AdvancedFieldSet {
   Access = 'access',
   Limits = 'limits',
   RetryPolicy = 'retryPolicy',
}

interface Props {
   disabled: boolean;
   name: string;
   readonly: boolean;
   sections: Set<AdvancedFieldSet>;
   toggleText: string;
   disabledRetryPolicyFields?: RetryPolicyFields[];
}

export const AdvancedSettingsSubForm: React.FC<Props> = React.memo(
   ({ name, toggleText, sections, readonly, disabled, disabledRetryPolicyFields = [] }) => {
      const form = useFormikContext();
      // undefined тут может быть потому что родительский тип в некоторых формах опциональный
      // (см. WorkloadDestroyCommand и т.д.)
      const { value } = form.getFieldMeta<WorkloadCommandWithAdvancedSettings | undefined>(name);

      return (
         <FormAdvancedSection
            header={null}
            toggleText={toggleText}
            bodyClassName={classes.advanced}
            openedByDefault={!isWorkloadAdvancedSettingsEmpty(value)}
         >
            {sections.has(AdvancedFieldSet.Access) ? (
               <AccessSubForm name={name} readonly={readonly} disabled={disabled} />
            ) : null}

            {sections.has(AdvancedFieldSet.Limits) ? (
               <LimitsSubForm name={name} readonly={readonly} disabled={disabled} />
            ) : null}

            {sections.has(AdvancedFieldSet.RetryPolicy) ? (
               <RetryPolicySubForm
                  name={name}
                  readonly={readonly}
                  disabled={disabled}
                  disabledFields={disabledRetryPolicyFields}
               />
            ) : null}
         </FormAdvancedSection>
      );
   },
);

AdvancedSettingsSubForm.displayName = 'AdvancedSettingsSubForm';
