import { modalService, useDismounted } from '@yandex-infracloud-ui/libs';
import React, { ReactNode, useCallback } from 'react';
import { takeUntil } from 'rxjs/operators';
import { Button } from '@yandex-cloud/uikit';

import { PerLocationStrategy } from '../../../models/ui';
import { StageLocationAction } from '../../../services/api/services/YpApi';
import { parseYpDatetime } from '../../../utils';
import { ApprovalList } from '../../common';

import { PerClusterApproveModal } from './PerClusterApproveModal/PerClusterApproveModal';
import { LocationApprovalStatus } from '../../../models/ui/stage/DeployUnit/DeployUnitStatusConverter';
import { useDeployUnitStatusView } from '../../../pages/stages/_stageId/status/hooks';

import classes from './PerClusterApprove.module.scss';

interface PerClusterApproveProps {
   stageId: string;
   duId: string;
   cluster: string;
}

export const PerClusterApprove: React.FC<PerClusterApproveProps> = React.memo(({ stageId, duId, cluster }) => {
   const { deployUnitStatusViewList } = useDeployUnitStatusView({ deployUnitSelectData: [{ stageId, duId }] });
   const [deployUnitStatusView = null] = deployUnitStatusViewList;

   const dismounted = useDismounted();
   const onActionClick = useCallback(
      (action: StageLocationAction) => {
         modalService
            .open(PerClusterApproveModal, {
               action,
               stageId,
               duId,
               duRevision: deployUnitStatusView?.spec?.revision ?? null,
               cluster,
            })
            .pipe(takeUntil(dismounted))
            .subscribe();
      },
      [stageId, duId, deployUnitStatusView?.spec?.revision, cluster, dismounted],
   );

   if (!deployUnitStatusView) {
      return null;
   }

   const { spec, overrides, locationApproves } = deployUnitStatusView;
   const perLocationSettings = {
      ...spec.perLocationSettings,
      ...(overrides?.deploySettings?.perLocationSettings ?? {}),
   };
   const { needApproval, strategy, locationOrder, isCustom } = perLocationSettings;
   const locationApprove = locationApproves.get(cluster);

   const needApprovalLocationOrder: string[] = locationOrder.filter(e => needApproval.has(e));

   // для параллельной выкатки доступны все, для последовательной начинаем с первого в списке для апрувов
   let enabled = needApprovalLocationOrder[0] === cluster || strategy === PerLocationStrategy.Parallel;

   if (locationApproves && strategy === PerLocationStrategy.Sequential) {
      const isCurrent = needApprovalLocationOrder.filter(e => !locationApproves.get(e))[0] === cluster;
      const canDisapprove = locationApprove?.status === LocationApprovalStatus.Approved;
      enabled = isCurrent || canDisapprove;
   }

   if (!isCustom) {
      return null;
   }

   const needApprovalCluster = needApproval.has(cluster);

   if (!needApprovalCluster) {
      return null;
   }

   let action: StageLocationAction | null = StageLocationAction.Approve;

   let approves: ReactNode;
   const { duRevision } = locationApprove ?? {};
   if (locationApprove && duRevision === deployUnitStatusView.spec.revision) {
      const {
         meta: { login, time },
         status,
      } = locationApprove;

      if (status === LocationApprovalStatus.Approved) {
         action = null;
      }

      if (login) {
         approves = (
            <div className={classes.locationApproves}>
               <ApprovalList
                  approvals={[
                     {
                        login,
                        status: status === LocationApprovalStatus.Approved ? 'approved' : 'disapproved',
                        timestamp: parseYpDatetime(time),
                     },
                  ]}
               />
            </div>
         );
      }
   }

   const actionText =
      action &&
      {
         [StageLocationAction.Approve]: 'Approve',
         [StageLocationAction.Disapprove]: 'Disapprove',
      }[action];

   return (
      <div className={classes.perClusterApprove}>
         {approves && <div>{approves}</div>}
         {action && (
            <div className={classes.action}>
               <Button
                  view={action === StageLocationAction.Approve ? 'action' : 'normal'}
                  className={classes['action-button']}
                  onClick={() => onActionClick(action as StageLocationAction)}
                  disabled={!enabled}
               >
                  {actionText}
               </Button>
            </div>
         )}
      </div>
   );
});

PerClusterApprove.displayName = 'PerClusterApprove';
