import React, { useCallback, useMemo, useState } from 'react';

import { DISMISS_REASON, IModalProps, ModalLayout } from '@yandex-infracloud-ui/libs';
import { Button } from '@yandex-cloud/uikit';
import { Form, Formik, FormikHelpers } from 'formik';
import { object, string } from 'yup';
import { useDispatch } from 'react-redux';

import classes from '../../../../design/commonModals.module.css';
import { DoStageLocationActionOptions, StageLocationAction } from '../../../../services/api/services/YpApi';
import { doStageLocationAction, getStages, useNetworkErrors } from '../../../../redux';
import { YpErrorTooltip } from '../../../network';
import { ActionState, useRequestControl } from '../../../../redux/hooks/useRequestControl';

const initialValue = {
   message: '',
};

export type StageLocationActionParams = typeof initialValue;

const ticketActionParamsValidationSchema = object<StageLocationActionParams>({
   message: string(),
});

type Props = IModalProps<StageLocationActionParams> & DoStageLocationActionOptions;

const stageActionRequestKey = 'stageLocationAction';
const requestKeys = [stageActionRequestKey];

export const PerClusterApproveModal: React.FC<Props> = React.memo(
   ({ action, stageId, duId, duRevision, cluster, ok, cancel }) => {
      const [formikHelpers, setFormikHelpers] = useState<FormikHelpers<StageLocationActionParams> | null>(null);

      const dispatch = useDispatch();

      const networkErrors = useNetworkErrors(requestKeys);
      const actionError = networkErrors[stageActionRequestKey];

      const doAction = useMemo(
         () =>
            doStageLocationAction.withRequestKey('stageLocationAction')({
               action,
               stageId,
               duId,
               duRevision,
               cluster,
            }),
         [action, stageId, duId, duRevision, cluster],
      );

      const breakSubmitting = useCallback(() => {
         if (formikHelpers) {
            formikHelpers.setSubmitting(false);
         }
      }, [formikHelpers]);

      const { activate: approveActionActivate, actionStates } = useRequestControl([stageActionRequestKey], {
         onSuccess: () => {
            dispatch(getStages({ objectIds: [stageId] }));
            ok({ message: '' });
            breakSubmitting();
         },
         onError: () => {
            breakSubmitting();
         },
      });

      const handleSubmit = useCallback(
         (v: StageLocationActionParams, helpers: FormikHelpers<StageLocationActionParams>) => {
            approveActionActivate(() => dispatch(doAction));
            setFormikHelpers(helpers);
         },
         [dispatch, doAction, approveActionActivate],
      );

      const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

      const actionText =
         action &&
         {
            [StageLocationAction.Approve]: 'Approve',
            [StageLocationAction.Disapprove]: 'Disapprove',
         }[action];

      const actionState = actionStates.get(stageActionRequestKey);
      const pending = actionState === ActionState.Pending;

      return (
         <ModalLayout
            title={`${actionText} for ${cluster.toUpperCase()}`}
            showFooter={false}
            onDismiss={dismiss}
            data-e2e={'Stage:ActionModal'}
         >
            <p style={{ maxWidth: '500px' }}>
               Current deploy unit <b>{duId}</b> with revision <b>{duRevision}</b> is deployed using per-location
               scenario. Please, confirm cluster <b>{cluster.toUpperCase()}</b> to continue. By the way, you can approve
               locations without waiting for the previous to finish completely.
            </p>

            <Formik
               initialValues={initialValue}
               onSubmit={handleSubmit}
               validationSchema={ticketActionParamsValidationSchema}
            >
               {form => (
                  <Form className={classes.form}>
                     <footer className={classes.footer}>
                        <Button view={'normal'} className={classes.button} onClick={dismiss} disabled={pending}>
                           Cancel
                        </Button>

                        <Button
                           view={'action'}
                           className={classes.button}
                           type={'submit'}
                           disabled={!form.isValid || pending}
                        >
                           {actionText}
                        </Button>
                     </footer>
                     {actionState === ActionState.Error && actionError?.error && (
                        <YpErrorTooltip error={actionError.error} request={actionError.request} />
                     )}
                  </Form>
               )}
            </Formik>
         </ModalLayout>
      );
   },
);

PerClusterApproveModal.displayName = 'PerClusterApproveModal';
