import { classNames, IPureValueProps, isEqual, styleHelpers, useDropdownDimensions } from '@yandex-infracloud-ui/libs';
import { Popup } from 'lego-on-react';
import React, { useCallback, useMemo, useRef, useState } from 'react';

import { Stage } from '../../../models/ui';
import { handleStopPropagation } from '../../../utils';
import { StagePart, StagePartType } from './models';

import classes from './SelectStagePart.module.css';
import { StagePartItem } from './StagePartItem';
import { StagePartItemView } from './StagePartItemView';

interface Props extends IPureValueProps<StagePart | null> {
   disabled?: boolean;
   stage: Stage;
   type: StagePartType;
   placeholder: string;
}

export const SelectStagePart: React.FC<Props> = React.memo(
   ({ type, stage, placeholder, value, onChange, disabled = false }) => {
      const [opened, setOpened] = useState(false);
      const stageParts: StagePart[] = useMemo(() => {
         const result: StagePart[] = [];

         result.push({ stage: stage.id, type: StagePartType.Stage });

         for (const du of stage.deployUnits) {
            result.push({ stage: stage.id, type: StagePartType.DeployUnit, deployUnit: du.id });

            for (const box of du.boxes) {
               result.push({ stage: stage.id, type: StagePartType.Box, deployUnit: du.id, box: box.id });

               for (const workload of box.workloads) {
                  result.push({
                     stage: stage.id,
                     type: StagePartType.Workload,
                     deployUnit: du.id,
                     box: box.id,
                     workload: workload.id,
                  });
               }
            }

            for (const resource of du.resources) {
               result.push({
                  stage: stage.id,
                  type: StagePartType.Resource,
                  deployUnit: du.id,
                  resource,
               });
            }
         }

         return result;
      }, [stage]);
      const switcherRef = useRef<HTMLDivElement>(null);
      const [width, maxHeight] = useDropdownDimensions(switcherRef as any);

      // region handlers
      const open = useCallback(() => {
         if (disabled) {
            return;
         }
         setOpened(true);
      }, [disabled]);

      const close = useCallback(() => {
         setOpened(false);
      }, []);

      const handleSelect = useCallback(
         (v: StagePart) => {
            close();
            onChange(v);
         },
         [close, onChange],
      );
      // endregion

      return (
         <div className={classNames({ [classes.disabled]: disabled })}>
            <div
               className={classes.toggle}
               onClick={open}
               ref={switcherRef}
               role={'button'}
               tabIndex={0}
               aria-label={'Open'}
            >
               {value ? (
                  <StagePartItemView className={classes.value} part={value} isTarget={true} withDeployUnit={true} />
               ) : (
                  <span className={classes.placeholder}>{placeholder}</span>
               )}
            </div>

            <Popup
               theme={'normal'}
               target={'anchor'}
               visible={opened}
               onOutsideClick={close}
               directions={['bottom-left', 'top-left']}
               anchor={switcherRef.current}
               cls={classes.popup}
            >
               <div style={{ minWidth: width, maxHeight, overflowY: 'auto' }} onScroll={handleStopPropagation}>
                  <ul className={styleHelpers.unstyledList}>
                     {stageParts.map((part, i) => (
                        /* eslint-disable react/no-array-index-key */
                        <StagePartItem
                           key={i}
                           onSelect={handleSelect}
                           part={part}
                           selected={isEqual(value, part)}
                           targetPartType={type}
                        />
                        /* eslint-enable react/no-array-index-key */
                     ))}
                  </ul>
               </div>
            </Popup>
         </div>
      );
   },
);

SelectStagePart.displayName = 'SelectStagePart';
