import React, { useMemo } from 'react';

import { Button, Loader } from '@yandex-cloud/uikit';
import { classNames } from '@yandex-infracloud-ui/libs';
import { format, fromUnixTime } from 'date-fns';

import { UserLink } from '../../lib';

import { Ants } from './__ants/Ants';
import { HistoryEvent } from './__historyEvent/HistoryEvent';
import classes from './History.module.css';
import { TimelineRow } from './models';
import { getStatusClassName } from './utils';

interface Props {
   columns: string[];
   eventRows: TimelineRow[];
   isFetching: boolean;
   canFetch: boolean;

   onLoadMore(): void;
}

export const History: React.FC<Props> = ({ columns, eventRows, isFetching, canFetch, onLoadMore }) => {
   const template = useMemo(() => `120px repeat(${columns.length}, 1fr)`, [columns.length]);

   const header = useMemo(
      () => (
         <div className={classNames(classes.tableHeader, classes.table)} style={{ gridTemplateColumns: template }}>
            <div className={classNames(classes.rowItem, classes.headerItem)}>Time</div>
            {columns.map(columnName => (
               <div key={`header_column_${columnName}`} className={classNames(classes.rowItem, classes.headerItem)}>
                  {columnName}
               </div>
            ))}
         </div>
      ),
      [columns, template],
   );

   const tableContent = useMemo(
      () =>
         eventRows.map(({ timestamp, events, user }, index) => {
            const date = fromUnixTime(timestamp);
            const dateStr = format(date, 'dd MMM yyyy, HH:mm:ss');

            const rowItems = [
               <div className={classes.rowItem}>
                  <div className={classes.rowContent}>
                     <div>{dateStr}</div>
                     <div>
                        <UserLink id={user} />
                     </div>
                  </div>
               </div>,
            ];

            events.forEach(({ content, isChanged, state, title, message }) => {
               const color = getStatusClassName(state, isChanged);

               rowItems.push(
                  <div className={classes.rowItem}>
                     <div
                        key={`row_${timestamp}_${columns[index]}`}
                        className={classNames(classes.rowContent, classes[color])}
                     >
                        <Ants isChanged={isChanged} className={classes.antsContainer} />
                        <HistoryEvent title={title} message={message} isChanged={isChanged} rows={content} />
                     </div>
                  </div>,
               );
            });

            return (
               <div key={timestamp} className={classes.table} style={{ gridTemplateColumns: template }}>
                  {rowItems}
               </div>
            );
         }),
      [columns, eventRows, template],
   );

   return (
      <div className={classes.history}>
         {header}
         <div className={classes.tableItems}>{tableContent}</div>
         <div className={classes.loaderContainer}>
            {isFetching && <Loader size={'s'} />}
            {canFetch && !isFetching && (
               <Button onClick={onLoadMore} view={'outlined'}>
                  Load more
               </Button>
            )}
         </div>
      </div>
   );
};
