import React from 'react';
import { Link } from 'react-router-dom';

import { urlBuilder } from '../../../models';
import { TDeployStatusDetails, TMultiClusterReplicaSet } from '../../../proto-typings';
import { RevisionPodsStatus } from '../RevisionPodsStatus/RevisionPodsStatus';
import { RevisionsProgressStatus } from '../RevisionsProgressStatus/RevisionsProgressStatus';

import classes from './MulticlusterReplicaSetPodsInfo.module.css';

const getDetailsMarkup = (details: TDeployStatusDetails | undefined, key: string) => {
   if (!details) {
      return null;
   }

   const { pods_total = 0, pods_ready = 0, pods_in_progress = 0, pods_failed = 0 } = details?.total_progress ?? {};
   const mappedRevisions = details.mapped_revisions;
   if (!mappedRevisions) {
      return null;
   }

   const revisionItems = Object.keys(mappedRevisions).map(revisionKey => {
      const value = mappedRevisions[revisionKey];
      return (
         <RevisionPodsStatus
            key={revisionKey}
            revision={Number(revisionKey)}
            failed={value?.pods_failed ?? 0}
            ready={value?.pods_ready ?? 0}
            inProgress={value?.pods_in_progress ?? 0}
            total={value?.pods_total ?? 0}
            className={classes.revision}
         />
      );
   });

   return (
      <div key={key} className={classes.podsSummary}>
         {pods_total !== 0 && (
            <RevisionsProgressStatus
               total={pods_total}
               failed={pods_failed}
               inProgress={pods_in_progress}
               ready={pods_ready}
               className={classes.progress}
            />
         )}
         {revisionItems.length > 0 && <div className={classes.revisionsList}>{revisionItems}</div>}
      </div>
   );
};

interface Props {
   replicaSet: TMultiClusterReplicaSet;
}

export const MulticlusterReplicaSetPodsInfo: React.FC<Props> = ({ replicaSet }) => {
   const { status } = replicaSet;
   const summary = getDetailsMarkup(status?.multi_cluster_deploy_status?.details, 'summary');

   const clusterDeployStatuses = status?.mapped_cluster_deploy_statuses ?? {};
   const perClusterSummary = Object.keys(clusterDeployStatuses).map(cluster => {
      const value = clusterDeployStatuses[cluster];
      const clusterDetails = value?.details;
      const podSetId = value?.pod_set_id ?? '';

      if (clusterDetails) {
         return (
            <div className={classes.podsSummary} key={cluster}>
               <div className={classes.podsTitle}>
                  <Link to={urlBuilder.ypPodSetPods(cluster, podSetId)}>{cluster.toUpperCase()} pods</Link>
               </div>
               {getDetailsMarkup(clusterDetails, cluster)}
            </div>
         );
      }

      return undefined;
   });

   return (
      <div className={classes.pods}>
         <div className={classes.podsTitle}>Pods summary</div>
         {summary}
         {perClusterSummary}
      </div>
   );
};
