import React, { useMemo } from 'react';

import { Link } from 'react-router-dom';
import { Table } from '@yandex-cloud/uikit';

import { formatBytes, formatCores, getPodSetService } from '../../../utils/yp';
import { columns, PodSetResourcesRowItem } from './models';
import { Resource } from './__resource/Reource';
import { TPodSet } from '../../../proto-typings';
import { YpLocation } from '../../../models/api';
import { urlBuilder } from '../../../models';
import { wrapCopy } from '../../lib';

import classes from './PodSetsResourcesTable.module.css';

interface Props {
   podSets: TPodSet[];
   cluster: YpLocation;
}

const getPodSetResourcesRow = (podSet: TPodSet, cluster: YpLocation): PodSetResourcesRowItem => {
   const { meta, status } = podSet;
   const { serviceLink, serviceId } = getPodSetService(podSet, cluster);
   const resourceUsage = status?.accounting?.resource_usage;

   const result: PodSetResourcesRowItem = {
      podSetId: meta?.id ? <Link to={urlBuilder.ypPodSet(cluster, meta.id)}>{meta.id}</Link> : undefined,
      serviceId: wrapCopy(serviceLink, serviceId),
      cpu: (
         <Resource
            value={resourceUsage?.cpu?.capacity && resourceUsage?.cpu?.capacity / 1000}
            format={formatCores}
            suffix={' vcores'}
         />
      ),
      memory: <Resource value={resourceUsage?.memory?.capacity} format={formatBytes} />,
      network: <Resource value={resourceUsage?.network?.bandwidth} format={formatBytes} suffix={'/s'} />,
      ipAddresses: <Resource value={resourceUsage?.internet_address?.capacity} />,
      hddCapacity: <Resource value={resourceUsage?.disk_per_storage_class?.hdd?.capacity} format={formatBytes} />,
      hddBandwidth: (
         <Resource value={resourceUsage?.disk_per_storage_class?.hdd?.bandwidth} format={formatBytes} suffix={'/s'} />
      ),
      ssdCapaciry: <Resource value={resourceUsage?.disk_per_storage_class?.ssd?.capacity} format={formatBytes} />,
      ssdBandwidth: (
         <Resource value={resourceUsage?.disk_per_storage_class?.ssd?.bandwidth} format={formatBytes} suffix={'/s'} />
      ),
   };

   if (resourceUsage?.gpu_per_model) {
      const models = Object.keys(resourceUsage.gpu_per_model);

      const gpuUsage = models
         .map(model => {
            const gpuModel = model as keyof typeof resourceUsage.disk_per_storage_class;

            const usage = resourceUsage.gpu_per_model[gpuModel];

            return { model: gpuModel, usage: usage.capacity };
         })
         .filter(x => x.usage !== 0)
         .map(({ model, usage }) => <Resource key={model} value={usage} title={model.toString()} />);

      if (gpuUsage.length) {
         result.gpu = <div className={classes.gpu}>{gpuUsage}</div>;
      }
   }

   return result;
};

export const PodSetsResourcesTable: React.FC<Props> = ({ podSets, cluster }) => {
   const podSetResourcesRows: PodSetResourcesRowItem[] = useMemo(
      () => podSets.map(podSet => getPodSetResourcesRow(podSet, cluster)),
      [podSets, cluster],
   );

   return <Table columns={columns} data={podSetResourcesRows} className={classes.table} />;
};
