import React, { useMemo } from 'react';

import { Table } from '@yandex-cloud/uikit';
import { Link } from 'react-router-dom';

import { columns, PodSetRowItem } from './models';
import { TPodSet } from '../../../proto-typings';
import { YpLocation } from '../../../models/api';
import { urlBuilder } from '../../../models';
import { getPodSetService } from '../../../utils/yp';
import { DeployEngine } from '../../../models/ui/yp/view';
import { wrapCopy } from '../../lib';

import classes from './PodSetsTable.module.css';

interface Props {
   podSets: TPodSet[];
   cluster: YpLocation;
   isFetching: boolean;
}

const getPodSetRow = (podSet: TPodSet, cluster: YpLocation): PodSetRowItem => {
   const { serviceLink, serviceId, deployEngine } = getPodSetService(podSet, cluster);
   const antiaffinityConstraints = podSet.spec?.antiaffinity_constraints;

   let constraintItems;
   if (antiaffinityConstraints) {
      constraintItems = antiaffinityConstraints.map(({ key, max_pods, pod_group_id_path }) => (
         <div className={classes.constraintsContainer} key={key}>
            <div className={classes.constraintsRow}>
               <span className={classes.constraintsTitle}>Key:</span>
               <span className={classes.constraintsContent}>{key}</span>
               <span className={classes.constraintsTitle}>Max pods:</span>
               <span className={classes.constraintsContent}>{max_pods}</span>
            </div>
            {pod_group_id_path && (
               <div className={classes.constraintsRow}>
                  <span className={classes.constraintsTitle}>Path:</span>
                  <span className={classes.constraintsContent}>{wrapCopy(pod_group_id_path)}</span>
               </div>
            )}
         </div>
      ));
   }

   return {
      podSetId: podSet.meta?.id ? (
         <Link to={urlBuilder.ypPodSet(cluster, podSet.meta.id)}>{podSet.meta.id}</Link>
      ) : undefined,
      serviceId: wrapCopy(serviceLink, serviceId),
      accountId: podSet.spec?.account_id ? wrapCopy(podSet.spec?.account_id) : undefined,
      nodeSegment: podSet.spec?.node_segment_id,
      nodeFilter: wrapCopy(podSet.spec?.node_filter),
      antiaffinityConstraints: constraintItems,
      deployEngine:
         deployEngine === DeployEngine.DeployRSC || deployEngine === DeployEngine.DeployMCRSC
            ? `Deploy/${deployEngine}`
            : deployEngine,
   };
};

export const PodSetsTable: React.FC<Props> = ({ podSets, cluster, isFetching }) => {
   const podSetRows = useMemo(() => podSets.map(podSet => getPodSetRow(podSet, cluster)), [podSets, cluster]);

   return (
      <Table
         columns={columns}
         data={podSetRows}
         className={classes.tableContainer}
         emptyMessage={isFetching ? 'Loading pod sets...' : undefined}
      />
   );
};

PodSetsTable.displayName = 'PodSetsTable';
