import { Table, TableColumnConfig, withTableSorting } from '@yandex-cloud/uikit';
import { ExternalLink, forHumanCapitalized, formatNumber } from '@yandex-infracloud-ui/libs';
import { format, formatDistanceToNowStrict } from 'date-fns';
import React, { useMemo, useState } from 'react';

import { Link } from 'react-router-dom';
import { EXTERNAL_LINKS, urlBuilder } from '../../../models';
import { YpLocation } from '../../../models/api';
import {
   DiskResource,
   EvictionStateMap,
   getResourcesByKind,
   SchedulingStateMap,
   StatusMap,
} from '../../../models/ui/yp/models';
import {
   EEvictionState,
   EPodCurrentState,
   EResourceKind,
   ESchedulingState,
   TPod,
   TResource,
} from '../../../proto-typings';
import { parseSimpleYpTimestamp } from '../../../utils';
import { formatBytes, formatCores } from '../../../utils/yp';
import { Status } from '../../lib';
import { ResourceBar } from '../ResourceBar/ResourceBar';

import { PodResourceUsage, PodUsageProps, TableRowItem } from './models';

import classes from './PodsTable.module.css';
import {
   commonFieldSort,
   compareRowItems,
   getCPUResourceUsage,
   getDiskResourceUsage,
   getMemoryResourceUsage,
   getNetworkResourceUsage,
} from './utils';

type PodsTableView = 'node' | 'cluster';

type TableRowComparer = (itemA: TableRowItem, itemB: TableRowItem) => number;

const TableWIthSorting = withTableSorting<TableRowItem>(Table);

const getSortingConfig = (id: string, view: PodsTableView): TableRowComparer | undefined => {
   if (view !== 'node') {
      return undefined;
   }

   switch (id) {
      case 'podId':
      case 'podSetId':
      case 'serviceOrStage':
      case 'schedulingLastUpdated':
      case 'state':
      case 'eviction':
      case 'scheduling':
         return commonFieldSort(id);

      case 'cpu':
         return compareRowItems(({ data }) => data.podResourceUsage.cpu.total);
      case 'memory':
         return compareRowItems(({ data }) => data.podResourceUsage.memory.total);
      case 'network':
         return compareRowItems(({ data }) => data.podResourceUsage.network.total);
      case 'hdd':
         return compareRowItems(({ data }) => data.podResourceUsage.hdd?.total);
      case 'hddBandwidth':
         return compareRowItems(({ data }) => data.podResourceUsage.hdd?.bandwidth);
      case 'ssd':
         return compareRowItems(({ data }) => data.podResourceUsage.ssd?.total);
      case 'ssdBandwidth':
         return compareRowItems(({ data }) => data.podResourceUsage.ssd?.bandwidth);
      default:
         return undefined;
   }
};

interface Props {
   pods: TPod[];
   nodeResources?: TResource[];
   cluster: YpLocation;
   view?: PodsTableView;
   emptyText?: string;
}

export const PodsTable: React.FC<Props> = ({ pods = [], nodeResources, cluster, view = 'node', emptyText }) => {
   const [podsUseHDD, setPodsUseHDD] = useState(false);
   const [podsUseSSD, setPodsUseSSD] = useState(false);

   const columns: TableColumnConfig<TableRowItem>[] = useMemo(() => {
      const items: (TableColumnConfig<TableRowItem> | false)[] = [
         {
            id: 'podId',
            name: 'Pod ID',
            sticky: 'left',
         },
         {
            id: 'podSetId',
            name: 'Pod set ID',
         },
         {
            id: 'serviceOrStage',
            name: 'Service',
         },
         {
            id: 'schedulingLastUpdated',
            name: 'Assigned',
         },
         {
            id: 'state',
            name: 'State',
         },
         {
            id: 'eviction',
            name: 'Eviction',
         },
         {
            id: 'scheduling',
            name: 'Scheduling',
         },
         {
            id: 'cpu',
            name: 'CPU',
         },
         {
            id: 'memory',
            name: 'Memory',
         },
         {
            id: 'network',
            name: 'Network',
         },
         podsUseHDD && {
            id: 'hdd',
            name: 'HDD capacity',
         },
         podsUseHDD && {
            id: 'hddBandwidth',
            name: 'HDD bandwidth',
         },
         podsUseSSD && {
            id: 'ssd',
            name: 'SSD capacity',
         },
         podsUseSSD && {
            id: 'ssdBandwidth',
            name: 'SSD bandwidth',
         },
         {
            id: 'cpuModel',
            name: 'CPU Model',
         },
      ];

      const filteredItems = items.filter((item): item is TableColumnConfig<TableRowItem> => Boolean(item));
      filteredItems.forEach(item => {
         item.meta = {
            sort: getSortingConfig(item.id, view),
         };
      });
      return filteredItems;
   }, [view, podsUseHDD, podsUseSSD]);

   const rows = pods.map(pod => {
      const podId = pod?.meta?.id ?? '';
      const podSetId = pod?.meta?.pod_set_id ?? 'unknown';
      const podResourceUsage: PodResourceUsage = {
         cpu: getCPUResourceUsage(pod?.spec?.resource_requests),
         memory: getMemoryResourceUsage(pod?.spec?.resource_requests),
         network: getNetworkResourceUsage(pod?.spec?.resource_requests),
      };

      const diskUsage = getDiskResourceUsage(pod.spec?.disk_volume_requests);
      if (diskUsage) {
         podResourceUsage.hdd = diskUsage.hdd;
         podResourceUsage.ssd = diskUsage.ssd;
      }

      if (podResourceUsage.hdd && !podsUseHDD) {
         setPodsUseHDD(true); // Warning! Сайд эффект. Не хочу лишний раз обходить все поды ради двух флагов
      }

      if (podResourceUsage.ssd && !podsUseSSD) {
         setPodsUseSSD(true); // Warning! Сайд эффект. Не хочу лишний раз обходить все поды ради двух флагов
      }

      let schedulingDate;
      let schedulingDateComponent;

      if (pod?.status?.scheduling?.last_updated) {
         schedulingDate = parseSimpleYpTimestamp(pod?.status?.scheduling?.last_updated);

         schedulingDateComponent = (
            <span>
               {format(schedulingDate, 'dd MMM yyyy, HH:mm')}
               <div className={classes.assignedDelta}>{formatDistanceToNowStrict(schedulingDate)} ago</div>
            </span>
         );
      }

      let serviceOrStage;
      let serviceOrStageLink;

      if (pod?.labels?.deploy?.stage_id) {
         serviceOrStage = pod?.labels?.deploy?.stage_id;
         serviceOrStageLink = (
            <Link to={urlBuilder.stage(pod?.labels?.deploy?.stage_id)}>{pod?.labels?.deploy?.stage_id}</Link>
         );
      } else {
         serviceOrStage = pod?.labels?.nanny_service_id;
         serviceOrStageLink = serviceOrStage ? (
            <ExternalLink href={EXTERNAL_LINKS.nannyService(pod?.labels?.nanny_service_id)}>
               {pod?.labels?.nanny_service_id}
            </ExternalLink>
         ) : (
            'Unknown'
         );
      }

      const state = pod?.status?.agent?.state ?? EPodCurrentState.PCS_UNKNOWN;
      const eviction = pod?.status?.eviction?.state ?? EEvictionState.ES_NONE;
      const scheduling = pod?.status?.scheduling?.state ?? ESchedulingState.SS_NONE;

      const row: TableRowItem = {
         data: {
            podId,
            podSetId,
            podResourceUsage,
            eviction,
            scheduling,
            schedulingLastUpdated: schedulingDate,
            state,
            serviceOrStage,
         },
         podId: <Link to={urlBuilder.ypPod(cluster, podId)}>{podId}</Link>,
         podSetId: <Link to={urlBuilder.ypPodSet(cluster, podSetId)}>{podSetId}</Link>,
         state: (
            <Status state={StatusMap.get(state)!} isAnimated={false}>
               {forHumanCapitalized(state)}
            </Status>
         ),
         eviction: (
            <Status state={EvictionStateMap.get(eviction)!} isAnimated={false}>
               {forHumanCapitalized(eviction)}
            </Status>
         ),
         scheduling: (
            <Status state={SchedulingStateMap.get(scheduling)!} isAnimated={false}>
               {forHumanCapitalized(scheduling)}
            </Status>
         ),
         schedulingLastUpdated: schedulingDateComponent,
         serviceOrStage: serviceOrStageLink,
         cpu: undefined,
         memory: undefined,
         network: undefined,
         hdd: undefined,
         hddBandwidth: undefined,
         ssd: undefined,
         ssdBandwidth: undefined,
         cpuModel: <div className={classes.cpuModel}>{pod?.labels?.node?.cpu_model}</div>,
      };

      if (view === 'node' && nodeResources) {
         const barsProps: PodUsageProps = {
            cpu: undefined,
            memory: undefined,
            network: undefined,
            hdd: undefined,
            hddBandwidth: undefined,
            ssd: undefined,
            ssdBandwidth: undefined,
         };

         if (podResourceUsage.cpu) {
            const nodeCPU = getResourcesByKind(nodeResources, EResourceKind.RK_CPU);
            if (nodeCPU) {
               const cpu = nodeCPU[0];
               barsProps.cpu = {
                  total: cpu.total / 1000,
                  used: podResourceUsage.cpu.total / 1000,
                  format: formatCores,
                  suffix: ' vcores',
               };
            }
         }

         if (podResourceUsage.memory) {
            const nodeMemory = getResourcesByKind(nodeResources, EResourceKind.RK_MEMORY);
            if (nodeMemory) {
               const memory = nodeMemory[0];
               barsProps.memory = {
                  total: memory.total,
                  used: podResourceUsage.memory.total,
                  format: formatBytes,
               };
            }
         }

         if (podResourceUsage.network) {
            const nodeNetwork = getResourcesByKind(nodeResources, EResourceKind.RK_NETWORK);
            if (nodeNetwork) {
               const network = nodeNetwork[0];
               barsProps.network = {
                  total: network.total,
                  used: podResourceUsage.network.total,
                  format: formatBytes,
                  suffix: '/s',
               };
            }
         }

         if (podResourceUsage.hdd) {
            const nodeDisk = getResourcesByKind(nodeResources, EResourceKind.RK_DISK) as DiskResource[];
            const nodeHDD = nodeDisk.find(x => x.storageClass === 'hdd');

            if (nodeHDD) {
               barsProps.hdd = {
                  total: nodeHDD.total,
                  used: podResourceUsage.hdd.total,
                  format: formatBytes,
               };

               barsProps.hddBandwidth = {
                  total: nodeHDD.bandwidthTotal,
                  used: podResourceUsage.hdd.bandwidth,
                  format: formatBytes,
                  suffix: '/s',
               };
            }
         }

         if (podResourceUsage.ssd) {
            const nodeDisk = getResourcesByKind(nodeResources, EResourceKind.RK_DISK) as DiskResource[];
            const nodeSSD = nodeDisk.find(x => x.storageClass === 'ssd');

            if (nodeSSD) {
               barsProps.ssd = {
                  total: nodeSSD.total,
                  used: podResourceUsage.ssd.total,
                  format: formatBytes,
               };

               barsProps.ssdBandwidth = {
                  total: nodeSSD.bandwidthTotal,
                  used: podResourceUsage.ssd.bandwidth,
                  format: formatBytes,
                  suffix: '/s',
               };
            }
         }

         row.cpu = barsProps.cpu && <ResourceBar {...barsProps.cpu} />;
         row.memory = barsProps.memory && <ResourceBar {...barsProps.memory} />;
         row.network = barsProps.network && <ResourceBar {...barsProps.network} />;
         row.hdd = barsProps.hdd && <ResourceBar {...barsProps.hdd} />;
         row.hddBandwidth = barsProps.hddBandwidth && <ResourceBar {...barsProps.hddBandwidth} />;
         row.ssd = barsProps.ssd && <ResourceBar {...barsProps.ssd} />;
         row.ssdBandwidth = barsProps.ssdBandwidth && <ResourceBar {...barsProps.ssdBandwidth} />;
      } else {
         row.cpu = podResourceUsage.cpu && `${formatNumber(podResourceUsage.cpu.total / 1000, formatCores)} vcores`;
         row.memory = podResourceUsage.memory && `${formatNumber(podResourceUsage.memory.total, formatBytes)}`;
         row.network = podResourceUsage.network && `${formatNumber(podResourceUsage.network.total, formatBytes)}/s`;
         row.hdd = podResourceUsage.hdd && `${formatNumber(podResourceUsage.hdd.total, formatBytes)}`;
         row.hddBandwidth = podResourceUsage.hdd && `${formatNumber(podResourceUsage.hdd?.bandwidth, formatBytes)}/s`;
         row.ssd = podResourceUsage.ssd && `${formatNumber(podResourceUsage.ssd.total, formatBytes)}`;
         row.ssdBandwidth = podResourceUsage.ssd && `${formatNumber(podResourceUsage.ssd.bandwidth, formatBytes)}/s`;
      }

      return row;
   });

   return <TableWIthSorting data={rows} columns={columns} className={classes.table} emptyMessage={emptyText} />;
};

PodsTable.displayName = 'PodsTable';
