import { EResourceKind, TPodSpec_TDiskVolumeRequest, TPodSpec_TResourceRequests } from '../../../proto-typings';
import { DiskResourceUsage, ResourceUsageBase, RowItemData, TableRowItem } from './models';

export const compareNumbers = (valueA: number | undefined, valueB: number | undefined) => {
   if (valueA === undefined && valueB !== undefined) {
      return -1;
   }

   if (valueA !== undefined && valueB === undefined) {
      return 1;
   }

   if (valueA === undefined && valueB === undefined) {
      return 0;
   }

   return valueA! - valueB!;
};

export const commonFieldSort = (fieldName: keyof RowItemData) => (itemA: TableRowItem, itemB: TableRowItem) => {
   const dataA = itemA.data[fieldName];
   const dataB = itemB.data[fieldName];

   if (dataA && !dataB) {
      return 1;
   }

   if (!dataA && dataB) {
      return -1;
   }

   if (!dataA && !dataB) {
      return 0;
   }

   switch (fieldName) {
      case 'schedulingLastUpdated':
         return (dataA as Date).getTime() - (dataB as Date).getTime();

      default:
         if (dataA!.toString() > dataB!.toString()) {
            return 1;
         }

         if (dataA!.toString() < dataB!.toString()) {
            return -1;
         }

         return 0;
   }
};

export const getDiskResourceUsage = (volumeRequests: TPodSpec_TDiskVolumeRequest[] | undefined) => {
   if (!volumeRequests) {
      return undefined;
   }

   const usageArray: DiskResourceUsage[] = volumeRequests.map(request => ({
      kind: EResourceKind.RK_DISK,
      total: request.quota_policy?.capacity ?? 0,
      storageClass: request.storage_class,
      bandwidth: request.quota_policy?.bandwidth_guarantee ?? 0,
   }));

   const resultObject: Record<string, DiskResourceUsage> = usageArray.reduce((val, current) => {
      const key = `${current.storageClass}`;
      const existingRecord = val[key];
      if (existingRecord) {
         existingRecord.total += current.total;
         existingRecord.bandwidth += current.bandwidth;
      } else {
         val[key] = current;
      }

      return val;
   }, {} as Record<string, DiskResourceUsage>);

   return resultObject;
};

export const getCPUResourceUsage = (
   resourceRequests: TPodSpec_TResourceRequests | undefined,
): ResourceUsageBase | undefined => {
   if (!resourceRequests) {
      return undefined;
   }

   return {
      kind: EResourceKind.RK_CPU,
      total: resourceRequests.vcpu_guarantee,
   };
};

export const getMemoryResourceUsage = (resourceRequests: TPodSpec_TResourceRequests | undefined) => {
   if (!resourceRequests) {
      return undefined;
   }

   return {
      kind: EResourceKind.RK_MEMORY,
      total: resourceRequests.memory_guarantee,
   };
};

export const getNetworkResourceUsage = (resourceRequests: TPodSpec_TResourceRequests | undefined) => {
   if (!resourceRequests) {
      return undefined;
   }

   return {
      kind: EResourceKind.RK_NETWORK,
      total: resourceRequests.network_bandwidth_guarantee,
   };
};

export const compareRowItems = (valueSelector: (item: any) => number | undefined) => (
   itemA: TableRowItem,
   itemB: TableRowItem,
): number => {
   const valueA = valueSelector(itemA);
   const valueB = valueSelector(itemB);

   return compareNumbers(valueA, valueB);
};
