import { Button, TextInput } from '@yandex-cloud/uikit';
import { YCSelect } from '@yandex-data-ui/common';
import { classNames, forHumanCapitalized } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useEffect, useMemo, useState } from 'react';

import { ReplicaSetFiltersParams, ReplicaSetState } from '../../../models/ui/yp/models';

import classes from './ReplicaSetFilters.module.css';

const stateSelectorItems = Object.values(ReplicaSetState).map(key => ({
   title: forHumanCapitalized(key),
   value: key,
}));

interface Props {
   filters: ReplicaSetFiltersParams;
   disabled: boolean;
   className?: string;

   onSubmit(params: ReplicaSetFiltersParams): void;
}

export const ReplicaSetFilters: React.FC<Props> = ({ onSubmit, filters, disabled, className }) => {
   const [filtersState, setFiltersState] = useState(filters);
   const state = useMemo(() => filtersState.state ?? [], [filtersState.state]);

   useEffect(() => {
      setFiltersState(filters);
   }, [filters]);

   const onInputChange = useCallback(
      <T extends keyof ReplicaSetFiltersParams>(fieldName: T, fieldValue: Required<ReplicaSetFiltersParams[T]>) => {
         const newFiltersState = { ...filtersState, [fieldName]: fieldValue };
         setFiltersState(newFiltersState);
      },
      [filtersState],
   );

   return (
      <div className={classNames(classes.filters, className)}>
         <div className={classes.row}>
            <TextInput
               value={filtersState.replicaSetId ?? ''}
               onUpdate={value => {
                  onInputChange('replicaSetId', value);
               }}
               placeholder={'Replica Set ID'}
               disabled={disabled}
            />
            <TextInput
               value={filtersState.stageId ?? ''}
               onUpdate={value => {
                  onInputChange('stageId', value);
               }}
               placeholder={'Stage ID'}
               disabled={disabled}
            />
            <YCSelect
               label={'State: '}
               type={YCSelect.MULTIPLE}
               items={stateSelectorItems}
               value={state}
               onUpdate={value => {
                  onInputChange('state', value as ReplicaSetState[]);
               }}
               disabled={disabled}
            />
            <Button
               onClick={() => {
                  onSubmit({});
               }}
               disabled={disabled}
            >
               Reset
            </Button>
            <Button
               onClick={() => {
                  onSubmit({ ...filtersState });
               }}
               disabled={disabled}
            >
               Search
            </Button>
         </div>
         <div className={classes.row}>
            <TextInput
               value={filtersState.query}
               onUpdate={value => {
                  onInputChange('query', value);
               }}
               placeholder={'Custom query'}
               disabled={disabled}
            />
         </div>
      </div>
   );
};
