import React, { ReactNode } from 'react';

import { forHumanCapitalized } from '@yandex-infracloud-ui/libs';
import { format, formatDistanceToNowStrict } from 'date-fns';
import { Link } from 'react-router-dom';

import { DefinitionList, DefinitionListItem } from '../../lib';
import { urlBuilder } from '../../../models';
import { EConditionStatus, TMultiClusterReplicaSet, TReplicaSet } from '../../../proto-typings';
import { ReplicaSetPodsInfo } from '../ReplicaSetPodsInfo/ReplicaSetPodsInfo';
import { MulticlusterReplicaSetPodsInfo } from '../MulticlusterReplicaSetPodsInfo/MulticlusterReplicaSetPodsInfo';
import { parseSimpleYpTimestamp } from '../../../utils';
import { InfoGroup } from '../InfoGroup/InfoGroup';

import classes from './ReplicaSetInfo.module.css';

interface Props {
   cluster: string;
   replicaSet: TReplicaSet | TMultiClusterReplicaSet;
   type: 'TReplicaSet' | 'TMultiClusterReplicaSet';
}

type KeyFilterCallback = (key: string) => boolean;

const objectToDefinitionListItems = (obj: any, keyFilter?: KeyFilterCallback): DefinitionListItem[] => {
   const items = Object.keys(obj)
      .filter(x => !keyFilter || keyFilter(x))
      .map(key => {
         const humanReadableName = key.split('_').map(forHumanCapitalized).join(' ');
         const value = obj[key];

         if (typeof value === 'object') {
            return objectToDefinitionListItems(value);
         }

         return {
            name: humanReadableName,
            content: value,
         };
      });

   return items.flat();
};

export const ReplicaSetInfo: React.FC<Props> = ({ replicaSet, cluster, type }) => {
   const { meta, spec, labels } = replicaSet;
   const replicaSetId = meta?.id ?? '';
   const [stageId, DUId] = replicaSetId.split('.');

   const fields: DefinitionListItem[] = [
      {
         name: 'Replica Set ID',
         content: replicaSetId,
         copyText: true,
      },
      {
         name: 'Stage ID',
         content: <Link to={urlBuilder.stageStatus(stageId, DUId)}>{stageId}</Link>,
         copyText: stageId,
      },
   ];

   if (type === 'TMultiClusterReplicaSet') {
      const rsClusters = (replicaSet as TMultiClusterReplicaSet)?.spec?.clusters ?? [];
      fields.push({
         name: 'Clusters',
         content: rsClusters.map(x => x.cluster.toUpperCase()).join(', '),
      });
   } else {
      fields.push({
         name: 'Pod Set ID',
         content: <Link to={urlBuilder.ypPodSet(cluster, replicaSetId)}>{replicaSetId}</Link>,
         copyText: replicaSetId,
      });
   }

   const deployStrategyItems: DefinitionListItem[] | undefined = spec?.deployment_strategy
      ? objectToDefinitionListItems(spec.deployment_strategy)
      : undefined;

   const ydLabelItems =
      labels && labels['yd.deploy_speed'] ? objectToDefinitionListItems(labels['yd.deploy_speed']) : undefined;

   const clusterErrors: ReactNode[] = [];

   if (type === 'TMultiClusterReplicaSet') {
      const mappedClusterDeployStatuses = (replicaSet as TMultiClusterReplicaSet).status
         ?.mapped_cluster_deploy_statuses;
      if (!mappedClusterDeployStatuses) {
         return null;
      }
      const statusItems: Record<string, DefinitionListItem[]> = {};
      Object.keys(mappedClusterDeployStatuses).forEach(key => {
         const value = mappedClusterDeployStatuses[key];
         const succeeded = value?.details?.controller_status?.last_attempt?.succeeded;

         if (succeeded) {
            const { last_transition_time, message, reason, status } = succeeded;

            if (status !== EConditionStatus.CS_TRUE) {
               const numberValue =
                  last_transition_time instanceof Date
                     ? last_transition_time!.getTime()
                     : last_transition_time!.seconds * 1000000 + last_transition_time!.nanos;

               const updateDate = parseSimpleYpTimestamp(numberValue);

               statusItems[key] = [
                  {
                     name: 'Status',
                     content: status,
                  },
                  {
                     name: 'Last transition time',
                     content: (
                        <span>
                           {format(updateDate, 'dd MMM yyyy, HH:mm')}
                           <div className={classes.delta}>{formatDistanceToNowStrict(updateDate)} ago</div>
                        </span>
                     ),
                  },
                  {
                     name: 'Reason',
                     content: reason,
                  },
                  {
                     name: 'Message',
                     content: message,
                  },
               ];
            }
         }
      });

      // eslint-disable-next-line
      for (const location in statusItems) {
         clusterErrors.push(
            <InfoGroup key={`controller_error_${location}`} title={location.toUpperCase()}>
               <DefinitionList items={statusItems[location]} />
            </InfoGroup>,
         );
      }
   }

   return (
      <div className={classes.container}>
         <div className={classes.info}>
            <InfoGroup>
               <DefinitionList items={fields} />
            </InfoGroup>

            {deployStrategyItems && (
               <InfoGroup title={'Deployment Strategy'}>
                  <DefinitionList items={deployStrategyItems} />
               </InfoGroup>
            )}

            {ydLabelItems && ydLabelItems.length > 0 && (
               <InfoGroup title={'yd.deploy_speed'}>
                  <DefinitionList items={ydLabelItems} />
               </InfoGroup>
            )}

            {clusterErrors.length !== 0 && <InfoGroup title={'Controller errors'}>{clusterErrors}</InfoGroup>}
         </div>
         {type === 'TReplicaSet' ? (
            <ReplicaSetPodsInfo replicaSet={replicaSet as TReplicaSet} />
         ) : (
            <MulticlusterReplicaSetPodsInfo replicaSet={replicaSet as TMultiClusterReplicaSet} />
         )}
      </div>
   );
};
