import React, { ReactNode, useMemo } from 'react';

import { Table, TableColumnConfig } from '@yandex-cloud/uikit';
import { classNames } from '@yandex-infracloud-ui/libs';
import { format, formatDistanceToNowStrict } from 'date-fns';
import { Link } from 'react-router-dom';

import { urlBuilder } from '../../../models';
import { ReplicaSetState, replicaSetStatusMap } from '../../../models/ui/yp/models';
import { EConditionStatus, TCondition, TMultiClusterReplicaSet, TReplicaSet } from '../../../proto-typings';
import { parseSimpleYpTimestamp } from '../../../utils';
import { Status, StatusState } from '../../lib';

import classes from './ReplicaSetsTable.module.css';

type ReplicaSetRowItem = {
   id: ReactNode;
   stageId: ReactNode;
   clusters: string;
   state: ReactNode;
   lastProcessTime: ReactNode;
   revision: string;
};

const columns: TableColumnConfig<ReplicaSetRowItem>[] = [
   {
      id: 'id',
      name: 'Replica Set ID',
      sticky: 'left',
   },
   {
      id: 'stageId',
      name: 'Stage ID',
   },
   {
      id: 'state',
      name: 'State',
   },
   {
      id: 'lastProcessTime',
      name: 'Last Process Time',
   },
   {
      id: 'revision',
      name: 'Revision',
   },
];

interface Props {
   replicaSets: TReplicaSet[] | TMultiClusterReplicaSet[];
   type: 'TReplicaSet' | 'TMulticlusterReplicaSet';
   className?: string;
   cluster: string;
}

export const ReplicaSetsTable: React.FC<Props> = ({ replicaSets, className, cluster, type }) => {
   const tableColumns = useMemo<TableColumnConfig<ReplicaSetRowItem>[]>(() => {
      if (type === 'TReplicaSet') {
         return columns;
      }

      const mcrsColumns = [...columns];
      mcrsColumns.splice(2, 0, {
         id: 'clusters',
         name: 'Clusters',
      });

      return mcrsColumns;
   }, [type]);

   const rows = useMemo<ReplicaSetRowItem[]>(
      () =>
         replicaSets.map((replicaSet: TReplicaSet | TMultiClusterReplicaSet) => {
            const id = replicaSet.meta?.id ?? '';
            const [stageId, DUId] = id.split('.');

            let state: ReplicaSetState = ReplicaSetState.IN_PROGRESS;
            let lastProcessTime: ReactNode;

            if (replicaSet.status) {
               const inProgress =
                  type === 'TReplicaSet'
                     ? (replicaSet as TReplicaSet).status?.in_progress_condition
                     : (replicaSet as TMultiClusterReplicaSet).status?.in_progress;
               const failed =
                  type === 'TReplicaSet'
                     ? (replicaSet as TReplicaSet).status?.failed_condition
                     : (replicaSet as TMultiClusterReplicaSet).status?.failed;
               const ready =
                  type === 'TReplicaSet'
                     ? (replicaSet as TReplicaSet).status?.ready_condition
                     : (replicaSet as TMultiClusterReplicaSet).status?.ready;

               if (inProgress?.status === EConditionStatus.CS_TRUE) {
                  state = ReplicaSetState.IN_PROGRESS;
               } else if (failed?.status === EConditionStatus.CS_TRUE) {
                  state = ReplicaSetState.FAILED;
               } else if (ready?.status === EConditionStatus.CS_TRUE) {
                  state = ReplicaSetState.READY;
               }

               const lastUpdateMap = new Map<number, Date>(
                  [inProgress, failed, ready]
                     .filter((condition): condition is TCondition => Boolean(condition?.last_transition_time))
                     .map(condition => {
                        const { last_transition_time } = condition;

                        const numberValue =
                           last_transition_time instanceof Date
                              ? last_transition_time!.getTime()
                              : last_transition_time!.seconds * 1000000 + last_transition_time!.nanos;

                        const updateDate = parseSimpleYpTimestamp(numberValue);

                        return [numberValue, updateDate];
                     }),
               );

               if (lastUpdateMap.size !== 0) {
                  const lastUpdateTimestamp = Math.max(...Array.from(lastUpdateMap.keys()));
                  const lastUpdate = lastUpdateMap.get(lastUpdateTimestamp)!;
                  lastProcessTime = (
                     <span>
                        {format(lastUpdate, 'dd MMM yyyy, HH:mm')}
                        <div className={classes.delta}>{formatDistanceToNowStrict(lastUpdate)} ago</div>
                     </span>
                  );
               }
            }

            let clusters = '';
            if (type === 'TMulticlusterReplicaSet') {
               const rsClusters = (replicaSet as TMultiClusterReplicaSet)?.spec?.clusters ?? [];
               clusters = rsClusters.map(x => x.cluster.toUpperCase()).join(', ');
            }

            const revision =
               (type === 'TReplicaSet'
                  ? (replicaSet as TReplicaSet).status?.revision_id
                  : (replicaSet as TMultiClusterReplicaSet).status?.revision.toString()) ?? '';

            return {
               id: (
                  <Link
                     to={
                        type === 'TReplicaSet'
                           ? urlBuilder.ypReplicaSet(cluster, id)
                           : urlBuilder.ypMulticlusterReplicaSet(id)
                     }
                  >
                     {id}
                  </Link>
               ),
               stageId: <Link to={urlBuilder.stageStatus(stageId, DUId)}>{stageId}</Link>,
               clusters,
               state: (
                  <Status isAnimated={false} state={replicaSetStatusMap.get(state) ?? StatusState.Unknown}>
                     {state}
                  </Status>
               ),
               lastProcessTime,
               revision,
            };
         }),
      [replicaSets, cluster, type],
   );

   return (
      <Table
         columns={tableColumns}
         data={rows}
         className={classNames(classes.table, className)}
         emptyMessage={'No items found'}
      />
   );
};
