import {
   EObjectType,
   TAccount,
   TApprovalPolicy,
   TDeployTicket,
   TMultiClusterReplicaSet,
   TNode,
   TNodeSegment,
   TPayload,
   TPerSegmentResourceTotals,
   TPod,
   TPodSet,
   TProject,
   TRelease,
   TReleaseRule,
   TReplicaSet,
   TReqCheckObjectPermissions,
   TReqCreateObject,
   TReqCreateObjects,
   TReqGetObjects,
   TReqGetUserAccessAllowedTo,
   TReqRemoveObject,
   TReqSelectObjectHistory,
   TReqSelectObjects,
   TReqUpdateObject,
   TResource,
   TRspCheckObjectPermissions,
   TRspCreateObject,
   TRspCreateObjects,
   TRspGetObjects,
   TRspGetUserAccessAllowedTo,
   TRspRemoveObject,
   TRspSelectObjectHistory,
   TRspSelectObjectHistory_TEvent,
   TRspSelectObjects,
   TRspUpdateObject,
   TStage,
   TStageDraft,
   TNetworkProject,
} from '../../../proto-typings';
import { NetworkRequest } from '../../../redux/slices/network/model';
import { ApiActionNames } from '../../../services/api/actions';
import { deepUpdateObject } from '../../../utils';
import { DeepPartial } from '../../typeHelpers';
import { ApiServiceName } from '../ApiServiceName';
import { ApiErrorItemWithData } from '../error';
import { ApiParams } from '../request';

export enum YpApiActions {
   CreateObject = 'CreateObject',
   CreateObjects = 'CreateObjects',
   GetObjects = 'GetObjects',
   RemoveObject = 'RemoveObject',
   SelectObjects = 'SelectObjects',
   UpdateObject = 'UpdateObject',
   SelectObjectHistory = 'SelectObjectHistory',
   CheckObjectPermissions = 'CheckObjectPermissions',
   GetUserAccessAllowedTo = 'GetUserAccessAllowedTo',
}

export interface YpApiParams {
   [YpApiActions.CreateObject]: ApiParams<DeepPartial<TReqCreateObject>, DeepPartial<TRspCreateObject>>;
   [YpApiActions.CreateObjects]: ApiParams<DeepPartial<TReqCreateObjects>, DeepPartial<TRspCreateObjects>>;
   [YpApiActions.GetObjects]: ApiParams<DeepPartial<TReqGetObjects>, DeepPartial<TRspGetObjects>>;
   [YpApiActions.RemoveObject]: ApiParams<DeepPartial<TReqRemoveObject>, DeepPartial<TRspRemoveObject>>;
   [YpApiActions.SelectObjects]: ApiParams<DeepPartial<TReqSelectObjects>, DeepPartial<TRspSelectObjects>>;
   [YpApiActions.UpdateObject]: ApiParams<DeepPartial<TReqUpdateObject>, DeepPartial<TRspUpdateObject>>;
   [YpApiActions.SelectObjectHistory]: ApiParams<
      DeepPartial<TReqSelectObjectHistory>,
      DeepPartial<TRspSelectObjectHistory>
   >;
   [YpApiActions.CheckObjectPermissions]: ApiParams<
      DeepPartial<TReqCheckObjectPermissions>,
      DeepPartial<TRspCheckObjectPermissions>
   >;
   [YpApiActions.GetUserAccessAllowedTo]: ApiParams<
      DeepPartial<TReqGetUserAccessAllowedTo>,
      DeepPartial<TRspGetUserAccessAllowedTo>
   >;
}

export enum YpLocation {
   XDC = 'xdc',
   SAS = 'sas',
   VLA = 'vla',
   MAN = 'man',
   IVA = 'iva',
   MYT = 'myt',
   SAS_TEST = 'sas-test',
   MAN_PRE = 'man-pre',
}

// improved TRspSelectObjects
export interface YpRspSelectObjects<T> {
   results: { values: [T] }[];
   timestamp: number;
   continuation_token: string;
}

export interface YpRspGetObjects<T> {
   subresponses: { result: { values: [T] } }[]; // sic!
   timestamp: number;
}

export type YpObjects = {
   [EObjectType.OT_ACCOUNT]: TAccount;
   [EObjectType.OT_PROJECT]: TProject;
   [EObjectType.OT_STAGE]: TStage;
   [EObjectType.OT_STAGE_DRAFT]: TStageDraft;
   [EObjectType.OT_DEPLOY_TICKET]: TDeployTicket;
   [EObjectType.OT_RELEASE]: TRelease;
   [EObjectType.OT_RELEASE_RULE]: TReleaseRule;
   [EObjectType.OT_APPROVAL_POLICY]: TApprovalPolicy;
   [EObjectType.OT_POD]: TPod;
   [EObjectType.OT_REPLICA_SET]: TReplicaSet;
   [EObjectType.OT_MULTI_CLUSTER_REPLICA_SET]: TMultiClusterReplicaSet;
   [EObjectType.OT_RELEASE]: TRelease;
   [EObjectType.OT_NODE]: TNode;
   [EObjectType.OT_NODE_SEGMENT]: TNodeSegment;
   [EObjectType.OT_RESOURCE]: TResource;
   [EObjectType.OT_POD_SET]: TPodSet;
   [EObjectType.OT_NETWORK_PROJECT]: TNetworkProject;
};

export type UsedYpObjectTypes = keyof YpObjects;
export type UsedYpObjects = YpObjects[UsedYpObjectTypes];

// пока не использовать!
// type part<T> = {
//    [K in keyof T]: YpObjectPart<T[K]>;
// };
// export type YpObjectPart<T> = T extends object ? part<T>[keyof T] | Partial<T> : T;

export type DeepNonNullable<T> = {
   [P in keyof T]: T[P] extends undefined
      ? true
      : T[P] extends Array<infer U>
      ? Array<DeepNonNullable<U>>
      : T[P] extends ReadonlyArray<infer U>
      ? ReadonlyArray<DeepNonNullable<U>>
      : T[P] extends Date | Function | Uint8Array
      ? T[P]
      : T[P] extends infer U | undefined
      ? DeepNonNullable<U>
      : T[P] extends object
      ? DeepNonNullable<T[P]>
      : NonNullable<T[P]>;
};

/**
 * типизация позволит подсказывать пути,
 * при использовании в апи тип T выводится автоматически, его не нужно указывать повторно
 * @example
 * (e: EObjectType.OT_PROJECT) => [e.meta.id, e.spec.account_id]
 *  // можно указать и обычные строки(для кастомных путей или несхематизированных данных)
 * (e: EObjectType.OT_DEPLOY_TICKET) => ['/control/commit']
 */
export type YpPaths<T extends UsedYpObjectTypes> = (ypObject: DeepNonNullable<YpObjects[T]>) => any[];

/**
 * Собирает строки для путей объекта yp апи
 * @param pathsFunc
 * @example
 * preparePaths((e: EObjectType.OT_PROJECT) => [e.meta.id, e.spec.account_id, '/annotations/project'])
 * // ['/meta/id', '/spec/account_id', '/annotations/project']
 */
export function prepareYpPaths<T extends UsedYpObjectTypes>(paths: YpPaths<T> | string[]): string[] {
   if (Array.isArray(paths)) {
      return paths;
   }

   const pathProxy = (value = '/') =>
      new Proxy(() => value, {
         get(target: () => string, name: string): () => string {
            return pathProxy(`${value}${value === '/' ? '' : '/'}${name}`);
         },
      });
   return paths(pathProxy() as any)
      .map(e => (typeof e === 'string' ? e : typeof e === 'function' ? e() : null))
      .filter(e => typeof e === 'string');
}

// переопределяем yson и все используемые типы, где он участвует
export type YpPayload = Omit<TPayload, 'yson'> & {
   yson: any;
};

/**
 * from TAttributeList, format for yson
 */
export interface YpAttributeList {
   value_payloads: YpPayload[];
   timestamps?: number[];
}

export type YpRspHistoryEvent = Omit<TRspSelectObjectHistory_TEvent, 'results'> & {
   results?: YpAttributeList;
};

export type YpRspSelectObjectHistory = Omit<TRspSelectObjectHistory, 'events'> & {
   events: YpRspHistoryEvent[];
};

export type YpHistoryEvent<T> = Omit<TRspSelectObjectHistory_TEvent, 'results'> & {
   results?: DeepPartial<T>;
};

export type YpSelectObjectHistory<T extends UsedYpObjects> = Omit<TRspSelectObjectHistory, 'events'> & {
   events: YpHistoryEvent<T>[];
};

export interface NetworkYpErrorData {
   ytError?: object;
   ytMessage?: string;
   ytTraceId?: string;
}

/**
 * превращает путь yp в массив имен полей
 */
export const getChainFromYpPath: (ypath: string) => string[] = ypath => {
   const yregex = /^(\/|(\/[a-z0-9-_]+)+)?$/;
   if (!yregex.test(ypath)) {
      throw new Error(`YPath must match the regex ${yregex}`);
   }
   if (ypath === '' || ypath === '/') {
      return [];
   }

   // '/meta/id' -> ['meta', 'id']
   return ypath.split('/').slice(1);
};

export function updateYpObject(object: object, paths: Array<string | symbol>, value: any) {
   const chain = paths.map(e => (typeof e === 'string' ? getChainFromYpPath(e) : e)).flat();
   return deepUpdateObject(object, chain, value);
}

// статистика по использованию путей
const usedYpPaths: {
   [type in EObjectType]?: {
      [action in YpApiActions]?: {
         [path: string]: boolean;
      };
   };
} = {};

export function getUsedYpPaths() {
   return usedYpPaths;
}

export function saveYpPaths<T extends UsedYpObjectTypes>(action: YpApiActions, type: T, paths: YpPaths<T> | string[]) {
   if (!(type in usedYpPaths)) {
      usedYpPaths[type] = {};
   }
   const actionsForType = usedYpPaths[type];
   if (!(action in actionsForType)) {
      actionsForType![action] = {};
   }
   for (const path of prepareYpPaths(paths)) {
      const pathsForType = actionsForType![action] as Record<string, boolean>;
      pathsForType[path as string] = true;
   }
}

window.getUsedYpPaths = getUsedYpPaths;

export function getYpDefaultErrorMessage(request: NetworkRequest): string | null {
   const { service, action, data } = request;

   if (!(service && service === ApiServiceName.YP && action && data && 'object_type' in data)) {
      return null;
   }
   const actionTitle =
      ({
         [YpApiActions.CreateObject]: 'Creating',
         [YpApiActions.UpdateObject]: 'Updating',
         [YpApiActions.RemoveObject]: 'Removing',
         [YpApiActions.GetObjects]: 'Fetching',
         [YpApiActions.SelectObjects]: 'Fetching',
      } as Partial<Record<ApiActionNames[ApiServiceName], string>>)[action] ?? null;
   return actionTitle ? `${actionTitle} ${(data as { object_type: UsedYpObjectTypes }).object_type}` : null;
}

export enum YtResponseCode {
   NON_EXIST = 100002,
}

export function isNonExistenceYpError(error: ApiErrorItemWithData<NetworkYpErrorData | string> | null): boolean {
   if (!error) {
      return false;
   }
   const { data } = error;
   if (!data || typeof data === 'string') {
      return false;
   }
   const ytError: Record<string, any> | undefined = data?.ytError;
   const ytErrorCode = ytError?.code;
   if (ytErrorCode && ytErrorCode === YtResponseCode.NON_EXIST) {
      return true;
   }
   return false;
}

export type YpAccountQuota = {
   accountId: number | 'tmp';
   serviceTitle: string;
   cluster: YpLocation;
   segment: string;
   resourceUsage: TPerSegmentResourceTotals;
};

export type AbcService = {
   id: number;
   name: {
      ru: string;
      en: string;
   };
};
