import { parseYtApiError, YtError } from './apiErrors';

describe('models|apiErrors', () => {
   it('should parseYtApiError shows root message', () => {
      const ytError = {
         message: 'root message',
         code: 3,
      };

      expect(parseYtApiError(ytError)).toEqual({
         longError: 'root message',
         shortError: 'root message',
         ytError,
      });
   });

   it('should parseYtApiError shows one inner message', () => {
      const ytError = {
         message: 'root message',
         code: 3,
         inner_errors: [
            {
               code: 4,
               message: 'inner message',
            },
         ],
      };

      expect(parseYtApiError(ytError)).toEqual({
         longError: 'root message: inner message',
         shortError: 'root message',
         ytError,
      });
   });

   it('should parseYtApiError shows several inner messages', () => {
      const ytError = {
         message: 'root message',
         code: 3,
         inner_errors: [
            {
               code: 4,
               message: 'inner message1',
            },
            {
               code: 4,
               message: 'inner message2',
            },
         ],
      };

      expect(parseYtApiError(ytError)).toEqual({
         longError: 'root message: inner message1, inner message2',
         shortError: 'root message',
         ytError,
      });
   });

   it('should parseYtApiError works with real data', () => {
      const ytError: YtError = {
         code: 1,
         message: 'Error validating "/spec" for release rule "khoden-deploy-ui-stage-rule"',
         attributes: {
            span_id: 2413756421268968400,
            fid: 18443122958498003000,
            trace_id: '32041399-66df1605-5524c9a3-b4021b51',
            tid: 10935258913669114000,
            datetime: '2020-03-28T20:54:57.879266Z',
            pid: 52,
            host: 'vla2-9104-066.xdc.yp.gencfg-c.yandex.net',
         },
         inner_errors: [
            {
               code: 1,
               message: 'Deploy patches map cannot be empty',
               attributes: {
                  span_id: 2413756421268968400,
                  fid: 18443122958498003000,
                  trace_id: '32041399-66df1605-5524c9a3-b4021b51',
                  tid: 10935258913669114000,
                  datetime: '2020-03-28T20:54:57.879230Z',
                  pid: 52,
                  host: 'vla2-9104-066.xdc.yp.gencfg-c.yandex.net',
               },
            },
         ],
      };

      expect(parseYtApiError(ytError)).toEqual({
         longError:
            'Error validating "/spec" for release rule "khoden-deploy-ui-stage-rule": Deploy patches map cannot be empty',
         shortError: 'Error validating "/spec" for release rule "khoden-deploy-ui-stage-rule"',
         ytError,
      });
   });

   it('should parseYtApiError with ytError', () => {
      const ytError = {
         code: 1,
         message: 'Error executing transactional request; transaction aborted',
         attributes: {
            host: 'man2-7465-91d.xdc.yp.gencfg-c.yandex.net',
            datetime: '2020-06-04T11:07:15.965021Z',
            pid: 2,
            tid: 4296989781397186600,
            fid: 18445776200059400000,
            trace_id: '55462987-b139122b-6f93546b-2c094f25',
            span_id: 3668585941901148000,
         },
         inner_errors: [
            {
               code: 100006,
               message:
                  'Access denied: "write" permission for stage "mtapi-production" is not allowed by any matching ACE',
               attributes: {
                  host: 'man2-7465-91d.xdc.yp.gencfg-c.yandex.net',
                  datetime: '2020-06-04T11:07:15.964936Z',
                  pid: 2,
                  tid: 4296989781397186600,
                  fid: 18445776200059400000,
                  trace_id: '55462987-b139122b-6f93546b-2c094f25',
                  span_id: 3668585941901148000,
                  permission: 'write',
                  object_type: 'stage',
                  attribute_path: '',
                  user_id: 'khoden',
                  object_id: 'mtapi-production',
               },
            },
         ],
      };

      expect(parseYtApiError({ ytError })).toEqual({
         longError:
            'Error executing transactional request; transaction aborted: Access denied: "write" permission for stage "mtapi-production" is not allowed by any matching ACE',
         shortError: 'Error executing transactional request; transaction aborted',
         ytError,
      });
   });
});
