import { toasts } from '@yandex-infracloud-ui/libs';
import { AxiosError } from 'axios';
import Notifications from '../old-code/services/Notifications';

interface ApiErrorAttributes {
   attribute_path?: string;
   datetime: string;
   fid: number;
   host: string;
   object_id?: string;
   object_type?: string;
   permission?: string;
   pid: number;
   span_id: number;
   tid: number;
   trace_id: string;
   user_id?: string;
}

interface ApiError {
   attributes?: ApiErrorAttributes;
   code: number;
   message: string;
}

interface YTErrorWrapper {
   ytError: YtError;
}

export interface YtError extends ApiError {
   inner_errors?: ApiError[];
}

export type AnyApiError = AxiosError | YtError | YTErrorWrapper;

interface ParsedYtError {
   longError: string;
   shortError: string;
   ytError: YtError;
}

export const parseYtApiError = (err: AnyApiError): ParsedYtError => {
   const axiosError = err as AxiosError;
   if (axiosError.isAxiosError) {
      return parseYtApiError(axiosError.response?.data);
   }

   const ytErrorWrapper = err as YTErrorWrapper;
   if ('ytError' in ytErrorWrapper) {
      return parseYtApiError(ytErrorWrapper.ytError);
   }

   const ytError = err as YtError;

   const shortError = ytError.message ?? ytError.toString();

   const innerMessages = (ytError.inner_errors ?? []).map(e => e.message);

   const longError = `${ytError.message}${innerMessages.length > 0 ? `: ${innerMessages.join(', ')}` : ''}`;

   return { longError, shortError, ytError };
};

export const apiError = (action: string, err: AxiosError, notifications?: Notifications): void => {
   const { longError, shortError: message, ytError } = parseYtApiError(err);

   if (!notifications) {
      toasts.apiError(action, longError);
      return;
   }

   notifications.add({
      level: 'error',
      title: `${action} error`,
      message,
      ytError,
      autoDismiss: 0,
   });
};

export const handleApiError = (action: string, notifications?: Notifications) => (err: AxiosError) =>
   apiError(action, err, notifications);
