/*
Возвращаем настройки сайдкаров деплой юнита.
Текущие версии сайдкаров (id sandbox ресурсов) входят в спеку DeployUnit по ключу `specPath`
При выходе новых версий новые id ресурсов прописываются per DeployUnit в labels.
Обновление на новые версии может происходить либо из формы, либо из StagePatcherOptionsModal
Проверка на необходимость обновления находится в StagePatcherOptionsModal
Патчинг лэйблов обновленных сайдкаров находится в StagePatcher
*/

import { ResourceGroup } from '../../modules/resources/config';
import { TDeployUnitSpec } from '../../proto-typings';
import { ConfirmationType } from './Confirmations';

// тип для ключей в DeployUnit.sidecars
export enum SidecarName {
   Logbroker = 'Logbroker',
   PodAgent = 'PodAgent',
   TVM = 'TVM',
   DynamicResource = 'DynamicResource',
   Juggler = 'Juggler',
   LogRotate = 'LogRotate',
   Coredump = 'Coredump',
}

export const SidecarsForUpdating = [
   SidecarName.Logbroker,
   SidecarName.PodAgent,
   SidecarName.TVM,
   SidecarName.DynamicResource,
] as const;

export type SidecarsForUpdatingType = typeof SidecarsForUpdating;
export type SidecarsForUpdatingUnion = SidecarsForUpdatingType[Exclude<keyof SidecarsForUpdatingType, keyof []>];

type DeployUnitSpecFields<T extends keyof TDeployUnitSpec> = T;
// тип для ключей в спеке deployUnit
export type SidecarsSpecPathsType = DeployUnitSpecFields<
   | 'logbroker_tools_sandbox_info'
   | 'pod_agent_sandbox_info'
   | 'tvm_sandbox_info'
   | 'dynamic_resource_updater_sandbox_info'
>;

export interface SidecarRecources {
   [ResourceGroup.Cpu]: number;
   [ResourceGroup.Mem]: number;
   [ResourceGroup.Disk]: number; // тип диска определяется позднее
}

export const emptySidecarResources: SidecarRecources = {
   [ResourceGroup.Cpu]: 0,
   [ResourceGroup.Mem]: 0,
   [ResourceGroup.Disk]: 0,
};

/**
 *
 */
export type SidecarsQuota = Record<
   SidecarName,
   {
      [patcherRevision: string]: SidecarRecources;
   }
>;

export interface SidecarType {
   title: string;
   confirmationType: ConfirmationType;
   labelKey: string;
   /** Текущие версии сайдкаров (id sandbox ресурсов) входят в спеку DeployUnit по ключу `specPath` */
   specPath: SidecarsSpecPathsType;
   patcherOption: string;
   sidecarSandboxId: string;
}

export type SidecarsConfigType = Record<SidecarsForUpdatingUnion, SidecarType>;

export const sidecarsUpdateConfig: SidecarsConfigType = {
   [SidecarName.Logbroker]: {
      title: 'Logbroker Tools',
      confirmationType: ConfirmationType.LogbrokerUpdate,
      labelKey: 'logbrokerToolsLayer',
      specPath: 'logbroker_tools_sandbox_info',
      patcherOption: 'updateLogbrokerTools',
      sidecarSandboxId: 'LOGBROKER_TOOLS_PORTO_LAYER',
   },
   [SidecarName.PodAgent]: {
      title: 'Pod Agent',
      confirmationType: ConfirmationType.PodAgentUpdate,
      labelKey: 'podBin',
      specPath: 'pod_agent_sandbox_info',
      patcherOption: 'updatePodAgent',
      sidecarSandboxId: 'POD_AGENT_BINARY',
   },
   [SidecarName.TVM]: {
      title: 'TVM Agent',
      confirmationType: ConfirmationType.TvmAgentUpdate,
      labelKey: 'tvm',
      specPath: 'tvm_sandbox_info',
      patcherOption: 'updateTvmAgent',
      sidecarSandboxId: 'PORTO_LAYER_DEPLOY_TVMTOOL',
   },
   [SidecarName.DynamicResource]: {
      title: 'DRU Agent',
      confirmationType: ConfirmationType.DruAgentUpdate,
      labelKey: 'druBin',
      specPath: 'dynamic_resource_updater_sandbox_info',
      patcherOption: 'updateDruAgent',
      sidecarSandboxId: 'DRU_LAYER',
   },
};
