import {
   TApprovalPolicySpec,
   TApprovalPolicySpec_EMode,
   TApprovalPolicySpec_TExclusion,
   TApprovalPolicySpec_TExclusion_EType,
} from '../../../proto-typings';

export enum ApprovalModeType {
   Optional = 'optional',
   Required = 'required',
}

export enum ApprovalPolicyType {
   Default = 'default',
   Mandatory = 'mandatory',
}

export interface ApprovalPolicySpec {
   mode: ApprovalModeType;
   approvalsCount: number;
   policy: ApprovalPolicyType;
   exclusions: {
      releaseRules: Set<string>;
   };
}

export interface ApprovalPermissions {
   default: boolean;
   mandatory: boolean;
   loginsPermissions: {
      [key: string]: {
         default: boolean;
         mandatory: boolean;
      };
   };
}

export function getMode(mode: TApprovalPolicySpec_EMode): ApprovalModeType {
   const { NONE, OPTIONAL, REQUIRED } = TApprovalPolicySpec_EMode;
   const { Optional, Required } = ApprovalModeType;
   return {
      [NONE]: Optional,
      [OPTIONAL]: Optional,
      [REQUIRED]: Required,
   }[mode];
}

export function getRawMode(mode: ApprovalModeType): TApprovalPolicySpec_EMode {
   const { OPTIONAL, REQUIRED } = TApprovalPolicySpec_EMode;
   const { Optional, Required } = ApprovalModeType;
   return {
      [Optional]: OPTIONAL,
      [Required]: REQUIRED,
   }[mode];
}

function getExclusions(exclusions: TApprovalPolicySpec_TExclusion[]): ApprovalPolicySpec['exclusions'] {
   const releaseRules = new Set<string>();
   if (exclusions) {
      for (const exclusion of exclusions) {
         const { type } = exclusion;
         if (type === TApprovalPolicySpec_TExclusion_EType.RELEASE_RULE) {
            const releaseRule = exclusion.release_rule;
            if (releaseRule) {
               const id = releaseRule.release_rule_id;
               releaseRules.add(id);
            }
         }
      }
   }
   return {
      releaseRules,
   };
}

function getRawExclusions(exclusions: ApprovalPolicySpec['exclusions']): Partial<TApprovalPolicySpec_TExclusion>[] {
   const result: Partial<TApprovalPolicySpec_TExclusion>[] = [];
   const { releaseRules } = exclusions;
   for (const ruleId of releaseRules) {
      result.push({
         type: TApprovalPolicySpec_TExclusion_EType.RELEASE_RULE,
         release_rule: {
            release_rule_id: ruleId,
         },
         compute_resources: undefined,
         environment_variable: undefined,
      });
   }
   return result;
}

export function getApprovalPolicySpec(rawSpec?: TApprovalPolicySpec): ApprovalPolicySpec | undefined {
   if (!rawSpec) {
      return undefined;
   }

   const { mode, multiple_approval, mandatory_multiple_approval, exclusions } = rawSpec;

   const policy = mandatory_multiple_approval ? ApprovalPolicyType.Mandatory : ApprovalPolicyType.Default;

   const count = {
      [ApprovalPolicyType.Default]: () => multiple_approval?.approvals_count ?? 0,
      [ApprovalPolicyType.Mandatory]: () => mandatory_multiple_approval?.approvals_count ?? 0,
   }[policy]();

   return {
      mode: getMode(mode),
      approvalsCount: count,
      policy,
      exclusions: getExclusions(exclusions),
   };
}

function getApprovalPolicyRawSpec(spec: ApprovalPolicySpec): TApprovalPolicySpec {
   const { mode, approvalsCount, policy, exclusions } = spec;
   // typescript не понимает, что это не произвольная строка, поэтому хак с unknown
   const approvalName = policy === ApprovalPolicyType.Mandatory ? 'mandatory_multiple_approval' : 'multiple_approval';
   return ({
      mode: getRawMode(mode),
      [approvalName]: {
         approvals_count: Number(approvalsCount),
      },
      exclusions: getRawExclusions(exclusions),
   } as unknown) as TApprovalPolicySpec;
}

export const ApprovalPolicyConverter = {
   getSpec: getApprovalPolicySpec,
   getRawSpec: getApprovalPolicyRawSpec,
};
