import { setAsArray } from '@yandex-infracloud-ui/libs';
import { boolean, number, object, Schema, string } from 'yup';

import { ApprovalModeType, ApprovalPolicySpec, ApprovalPolicyType } from './ApprovalPolicy';

export interface ApprovalPolicyFormParams {
   activeMode: boolean;
   count: number;
   mandatoryPolicy: ApprovalPolicyType;
   exclusions: {
      releaseRules: Set<string>;
   };
}

export const approvalPolicyValidationSchema = object<ApprovalPolicyFormParams>({
   activeMode: boolean(),
   count: number().label('Count').required().integer().positive(),
   mandatoryPolicy: string().oneOf([
      ApprovalPolicyType.Default,
      ApprovalPolicyType.Mandatory,
   ]) as Schema<ApprovalPolicyType>,
   exclusions: object<ApprovalPolicyFormParams['exclusions']>({
      releaseRules: setAsArray().of(string()) as any,
   }),
});

function getSpecFromValues(formValues: ApprovalPolicyFormParams): ApprovalPolicySpec {
   const { activeMode, count, mandatoryPolicy, exclusions } = formValues;

   return {
      mode: activeMode ? ApprovalModeType.Required : ApprovalModeType.Optional,
      approvalsCount: Number(count),
      policy: mandatoryPolicy,
      exclusions,
   };
}

function getValuesFromSpec(spec?: ApprovalPolicySpec): ApprovalPolicyFormParams {
   const {
      mode = ApprovalModeType.Required,
      approvalsCount = 0,
      policy = ApprovalPolicyType.Default,
      exclusions = {
         releaseRules: new Set<string>(),
      },
   } = spec ?? {};
   return {
      activeMode: mode === ApprovalModeType.Required,
      count: approvalsCount,
      mandatoryPolicy: policy,
      exclusions,
   };
}

export const ApprovalPolicyFormConverter = {
   getSpecFromValues,
   getValuesFromSpec,
};
