export type GetNormalizeListOptions<T, K extends string> = {
   selectId(elem: T): string;
   groupBy?: Record<K, (elem: T) => string>;
};

export type GetNormalizeListResult<T, K extends string> = {
   idList: string[];

   /** Значения списка */
   values: ReadonlyMap<string, T>;

   /**
    * группы id по выбранным признакам
    */
   groupBy: Record<K, ReadonlyMap<string, ReadonlySet<T>>>;
};

export function getNormalizeList<T, K extends string>({
   selectId,
   groupBy,
}: GetNormalizeListOptions<T, K>): (options: { list?: T[] }) => GetNormalizeListResult<T, K> {
   return ({ list = [] }) => {
      const idList = list.map(selectId);
      const values = new Map(list.map(elem => [selectId(elem), elem]));

      const groups: Partial<Record<K, Map<string, Set<T>>>> = {};
      for (const [groupName, select] of Object.entries<(elem: T) => string>(groupBy ?? {})) {
         if (!groups.hasOwnProperty(groupName)) {
            groups[groupName as K] = new Map();
         }
         for (const elem of list) {
            const group = groups[groupName as K]!;
            const groupId = select(elem);
            if (!group.has(groupId)) {
               group.set(groupId, new Set());
            }
            group.get(groupId)!.add(elem);
         }
      }

      return {
         idList,
         values,
         groupBy: groups as GetNormalizeListResult<T, K>['groupBy'],
      };
   };
}
