import { ResourceGroup } from '../../../modules/resources/config';
import { TResourceTotals } from '../../../proto-typings';
import { buildTree, createKey } from '../../../utils';

import { defaultQuotaOrder, QuotaKeyParams, QuotaMetric, QuotaSegmentSet } from './model';

function addRecord(
   store: Record<string, number>,
   params: Omit<QuotaKeyParams, 'resource' | 'resourceGroup'>,
   resourceGroup: ResourceGroup,
   resource: string,
   value: number,
) {
   const resourceParams: QuotaKeyParams = { ...params, resourceGroup, resource };
   store[createKey((resourceParams as unknown) as Record<string, string>)] = value;
}

function getUIModel({
   location,
   abc,
   metric,
   resourcesData,
}: {
   location: string;
   abc: string;
   metric: QuotaMetric;
   resourcesData: Partial<TResourceTotals>;
}): Record<string, number> {
   const store: Record<string, number> = {};
   const dataPerSegment = resourcesData.per_segment;
   if (!dataPerSegment) {
      return store;
   }
   for (const segment of Object.keys(dataPerSegment)) {
      if (QuotaSegmentSet.has(segment)) {
         const params: Omit<QuotaKeyParams, 'resource' | 'resourceGroup'> = { location, segment, abc, metric };

         const addResourceRecord = (resourceGroup: ResourceGroup, resource: string, value: number) =>
            addRecord(store, params, resourceGroup, resource, value);

         const resources = dataPerSegment[segment];

         const { cpu, memory, disk_per_storage_class, internet_address, gpu_per_model, network } = resources;

         if (cpu) {
            addResourceRecord(ResourceGroup.Cpu, '', cpu.capacity);
         }

         if (memory) {
            addResourceRecord(ResourceGroup.Mem, '', memory.capacity);
         }

         if (disk_per_storage_class) {
            for (const diskType of Object.keys(disk_per_storage_class)) {
               const diskData = disk_per_storage_class[diskType];

               addResourceRecord(ResourceGroup.Disk, diskType, diskData.capacity);
               addResourceRecord(ResourceGroup.DiskBandwidth, diskType, diskData.bandwidth);
            }
         }

         if (network) {
            addResourceRecord(ResourceGroup.NetworkBandwidth, '', network.bandwidth);
         }

         if (internet_address) {
            addResourceRecord(ResourceGroup.InternetAddress, '', internet_address.capacity);
         }

         if (gpu_per_model) {
            for (const model of Object.keys(gpu_per_model)) {
               const gpuData = gpu_per_model[model];
               addResourceRecord(ResourceGroup.Gpu, model, gpuData.capacity);
            }
         }
      }
   }

   return store;
}

type List<T extends readonly [string, ...string[]]> = T extends readonly [infer X, ...infer Y] ? [X, ...Y] : never;

// FIXME: построить честный порядок
function getTree<T extends [keyof QuotaKeyParams, ...(keyof QuotaKeyParams)[]]>(
   store: Record<string, number>,
   order: T,
) {
   const orderSet = new Set(order);
   const fullOrder = order
      .filter((e, i, list) => list.slice(0, i).every(t => t !== e))
      .concat(defaultQuotaOrder.filter(e => !orderSet.has(e)));
   return buildTree(store, (fullOrder as unknown) as List<typeof defaultQuotaOrder>);
}

export const quotaConverter = {
   getUIModel,
   getTree,
};
