import { IJsonable } from '@yandex-infracloud-ui/libs';

import {
   EConditionStatus,
   EDeployPatchActionType,
   TApproval_EApprovalStatus,
   TApproval_TUserApproval_EUserApprovalStatus,
   TDeployPatchSpec,
   TDeployPatchStatus,
   TDeployTicket,
   TDeployTicketSpec_EDeployTicketSourceType,
} from '../../../../proto-typings';
import { Entity } from '../../../../redux/models';
import { parseYpDatetime, ProtoDate } from '../../../../utils';

import { ReleaseRuleType } from '../releaseRule/ReleaseRuleFormParams';
import { TicketStatus } from './statuses';

export interface DeployPatch {
   /**
    * Для dynamic ресурсов нерелевантен (пока), так что будет null
    */
   deployUnitId: string | null;
   id: string;
   status: TicketStatus;
}

interface ApproveData {
   comment: string | null;
   date: number | null;
}
export interface TicketApproval {
   approve: Set<string>;
   disapprove: Set<string>;
   data: Map<string, ApproveData>;
   status: TApproval_EApprovalStatus;
}

export interface DeployTicket {
   creationDate: number;
   executionEnd: number | null;
   executionStart: number | null;
   id: string;
   patches: DeployPatch[];
   releaseId: string;
   releaseRuleId: string;
   stageId: string;
   status: TicketStatus;
   approval: TicketApproval;
   sourceType: TDeployTicketSpec_EDeployTicketSourceType;
   stageDraftId: string;
   stageDraftRevision: number;
   title: string;
   description: string;
}

export class DeployTicketConverter implements Entity, DeployTicket, IJsonable {
   public static fromApi(raw: TDeployTicket): DeployTicket {
      return new DeployTicketConverter(raw).toJSON();
   }

   private static getPatchStatus(patchStatus: TDeployPatchStatus): TicketStatus {
      if (
         !patchStatus.action ||
         !patchStatus.action.type ||
         patchStatus.action.type === EDeployPatchActionType.DPAT_NONE
      ) {
         return TicketStatus.WaitingForCommit;
      }

      switch (patchStatus.action.type) {
         case EDeployPatchActionType.DPAT_COMMIT: {
            // При этом действии смотрим уже на progress
            const { progress } = patchStatus;
            if (!progress) {
               return TicketStatus.Committed;
            }

            const map = [
               [TicketStatus.Pending, progress.pending],
               [TicketStatus.InProgress, progress.in_progress],
               [TicketStatus.Success, progress.success],
               [TicketStatus.Failed, progress.failed],
               [TicketStatus.Cancelled, progress.cancelled],
               [TicketStatus.Closed, progress.closed], // TODO update porto-typings!!! DEPLOY-2138
            ] as const;

            for (const [status, condition] of map) {
               if (condition?.status === EConditionStatus.CS_TRUE) {
                  return status;
               }
            }
            break;
         }
         case EDeployPatchActionType.DPAT_ON_HOLD: {
            return TicketStatus.OnHold;
         }
         case EDeployPatchActionType.DPAT_SKIP: {
            return TicketStatus.Skip;
         }
         case EDeployPatchActionType.DPAT_WAIT: {
            return TicketStatus.Wait;
         }
      }

      console.warn('Unsupported patch status', patchStatus.action.type);
      return TicketStatus.Unknown;
   }

   private static getPatchType(rawPatch: TDeployPatchSpec): ReleaseRuleType | null {
      if (rawPatch.sandbox) {
         return ReleaseRuleType.Sandbox;
      }

      if (rawPatch.docker) {
         return ReleaseRuleType.Docker;
      }

      return null;
   }

   // TODO вынести в parsers
   private static parseDate(v: Date | ProtoDate | undefined): number | null {
      if (v instanceof Date) {
         return v.getTime();
      }

      const date = parseYpDatetime(v);

      return date === null ? null : date.getTime();
   }

   public static getApproveLogins(deployTickets: DeployTicket[]): Set<string> {
      return new Set(
         deployTickets
            .filter(ticket => ticket.status === TicketStatus.WaitingForCommit)
            .flatMap(ticket => [...(ticket.approval?.data?.keys() ?? [])]),
      );
   }

   public readonly creationDate: number;

   public readonly executionEnd: number | null;

   public readonly executionStart: number | null;

   public readonly id: string;

   public readonly sourceType: TDeployTicketSpec_EDeployTicketSourceType;

   public readonly patches: DeployPatch[];

   public readonly releaseId: string;

   public readonly releaseRuleId: string;

   public readonly stageDraftId: string;

   public readonly stageDraftRevision: number;

   public readonly title: string;

   public readonly description: string;

   public readonly stageId: string;

   public readonly status: TicketStatus;

   public readonly approval: TicketApproval;

   constructor(private raw: TDeployTicket) {
      this.id = this.raw.meta!.id!;
      this.sourceType = this.raw.spec!.source_type || TDeployTicketSpec_EDeployTicketSourceType.RELEASE_INTEGRATION;
      this.releaseId = this.raw.spec!.release_id;
      this.releaseRuleId = this.raw.spec!.release_rule_id || '';
      this.creationDate = this.raw.meta!.creation_time! / 1000;
      this.stageId = this.raw.meta!.stage_id;

      this.stageDraftId = this.raw.spec!.stage_draft_id;
      this.stageDraftRevision = this.raw.spec!.stage_draft_revision;

      this.executionStart = DeployTicketConverter.parseDate(this.raw.status?.progress?.start_time);
      this.executionEnd = DeployTicketConverter.parseDate(this.raw.status?.progress?.end_time);

      this.title = this.raw.spec!.title || '';
      this.description = this.raw.spec!.description || '';

      if (this.sourceType === TDeployTicketSpec_EDeployTicketSourceType.RELEASE_INTEGRATION) {
         this.patches = this.getPatches();
      } else {
         this.patches = [];
      }
      this.status = this.getTicketStatus();
      this.approval = this.getApproval();
   }

   public toJSON(): DeployTicket {
      return {
         creationDate: this.creationDate,
         executionEnd: this.executionEnd,
         executionStart: this.executionStart,
         id: this.id,
         patches: this.patches,
         releaseId: this.releaseId,
         releaseRuleId: this.releaseRuleId,
         stageId: this.stageId,
         status: this.status,
         approval: this.approval,
         sourceType: this.sourceType,
         stageDraftId: this.stageDraftId,
         stageDraftRevision: this.stageDraftRevision,
         title: this.title,
         description: this.description,
      };
   }

   private getApproval(): TicketApproval {
      const rawApproval = this.raw.status?.approval;
      const approval: TicketApproval = {
         approve: new Set(),
         disapprove: new Set(),
         data: new Map(),
         status: rawApproval?.status ?? TApproval_EApprovalStatus.NONE,
      };
      for (const item of rawApproval?.user_approvals ?? []) {
         const { login, status, comment, approval_time } = item;
         if (status === TApproval_TUserApproval_EUserApprovalStatus.APPROVED) {
            approval.approve.add(login);
         } else if (status === TApproval_TUserApproval_EUserApprovalStatus.DISAPPROVED) {
            approval.disapprove.add(login);
         }
         approval.data.set(login, { comment, date: DeployTicketConverter.parseDate(approval_time) });
      }
      return approval;
   }

   private getPatches() {
      return Object.keys(this.raw.spec!.patches).map(patchId => {
         const rawPatch = this.raw.spec!.patches[patchId]!;
         const type = DeployTicketConverter.getPatchType(rawPatch);
         const status = DeployTicketConverter.getPatchStatus(this.raw.status!.patches[patchId]!);

         let deployUnitId: string | null = null;
         switch (type) {
            case ReleaseRuleType.Docker:
               deployUnitId = rawPatch.docker!.docker_image_ref?.deploy_unit_id ?? null;
               break;

            case ReleaseRuleType.Sandbox:
               deployUnitId = rawPatch.sandbox!.static?.deploy_unit_id ?? null;
         }

         return { deployUnitId, id: patchId, status } as DeployPatch;
      });
   }

   /**
    * Практически копия _getPatchStatus, только меньше статусов для action='commit'
    */
   private getTicketStatus(): TicketStatus {
      const ticketStatus = this.raw.status!;
      if (
         !ticketStatus.action ||
         !ticketStatus.action.type ||
         ticketStatus.action.type === EDeployPatchActionType.DPAT_NONE
      ) {
         return TicketStatus.WaitingForCommit;
      }

      switch (ticketStatus.action.type) {
         case EDeployPatchActionType.DPAT_COMMIT: {
            // При этом действии смотрим уже на progress
            const { progress } = ticketStatus;
            if (!progress) {
               return TicketStatus.Committed;
            }

            const map = [
               [TicketStatus.Pending, progress.pending],
               [TicketStatus.InProgress, progress.in_progress],
               [TicketStatus.Closed, progress.closed], // TODO update porto-typings!!! DEPLOY-2138
            ] as const;

            for (const [status, condition] of map) {
               if (condition?.status === EConditionStatus.CS_TRUE) {
                  return status;
               }
            }
            break;
         }
         case EDeployPatchActionType.DPAT_ON_HOLD: {
            return TicketStatus.OnHold;
         }
         case EDeployPatchActionType.DPAT_SKIP: {
            return TicketStatus.Skip;
         }
         case EDeployPatchActionType.DPAT_WAIT: {
            return TicketStatus.Wait;
         }
      }

      console.warn('Unsupported ticket status', ticketStatus.action.type);
      return TicketStatus.Unknown;
   }
}

export type OptionalDeployTicketColumn = 'stage' | 'release';
