import { TApproval_EApprovalStatus } from '../../../../proto-typings';
import { DeployTicket } from './DeployTicket';
import { ApprovalModeType, ApprovalPolicySpec, ApprovalPolicyType } from '../../approvalPolicy/ApprovalPolicy';
import { TicketStatus } from './statuses';
import { createKey } from '../../../../utils';
import { ApprovalPermissionsResult } from '../../../../redux/hooks/useApprovalPermissions';
import { TicketAction } from './actions';

const ApprovalStatus = TApproval_EApprovalStatus;

export function isApprovalRequired(ticket: DeployTicket, approvalPolicySpec: ApprovalPolicySpec | null): boolean {
   const ruleExclusions = approvalPolicySpec?.exclusions?.releaseRules;
   const isExclusion = ruleExclusions ? ruleExclusions.has(ticket.releaseRuleId) : false;
   return approvalPolicySpec?.mode === ApprovalModeType.Required && !isExclusion;
}

/**
 * Доступные действия над деплойным тикетом с учётом логина текущего пользователя.
 * Используется для показа действий.
 */
export function getDeployTicketActionLoginAvailable({
   ticket,
   login,
   approvalPolicySpec,
   approvalPermissions,
}: {
   ticket: DeployTicket;
   login: string;
   approvalPolicySpec: ApprovalPolicySpec | null;
   approvalPermissions: ApprovalPermissionsResult;
}): Set<TicketAction> {
   const available: Set<TicketAction> = new Set([TicketAction.Skip]);

   const approvalRequired = isApprovalRequired(ticket, approvalPolicySpec);
   const { left: approvalsLeftCount } = getApprovalsCount({ ticket, approvalPermissions, approvalPolicySpec });

   const commitAvailable = !approvalRequired || approvalsLeftCount === 0;
   if (commitAvailable) {
      available.add(TicketAction.Commit);
   }

   const { approval, stageId } = ticket;

   // для текущего пользователя
   const approveActionsAvailable =
      (approvalPermissions[createKey({ login: '', stageId, type: 'default' })] ||
         approvalPermissions[createKey({ login: '', stageId, type: 'mandatory' })]) &&
      approvalRequired;

   const approveAvailable = approveActionsAvailable && !approval.approve.has(login);
   if (approveAvailable) {
      available.add(TicketAction.Approve);
   }

   const disapproveAvailable = approveActionsAvailable && !approval.disapprove.has(login);
   if (disapproveAvailable) {
      available.add(TicketAction.Disapprove);
   }

   return available;
}

/**
 * Разрешенные действия для деплойного тикета без учёта текущего логина.
 * Используется для показа статуса тикета.
 */
export function getDeployTicketActionAvailable({
   ticket,
   approvalPolicySpec,
   approvalPermissions,
}: {
   ticket: DeployTicket;
   approvalPolicySpec: ApprovalPolicySpec | null;
   approvalPermissions: ApprovalPermissionsResult;
}): Set<TicketAction> {
   const available: Set<TicketAction> = new Set([TicketAction.Skip]);

   const approvalRequired = isApprovalRequired(ticket, approvalPolicySpec);
   const { left: approvalsLeftCount } = getApprovalsCount({ ticket, approvalPermissions, approvalPolicySpec });

   const commitAvailable = !approvalRequired || approvalsLeftCount === 0;
   if (commitAvailable) {
      available.add(TicketAction.Commit);
   }

   if (approvalRequired) {
      available.add(TicketAction.Disapprove);
      available.add(TicketAction.Approve);
   }

   return available;
}

interface ApprovalsCount {
   needed: number;
   sum: number;
   left: number;
   mandatorySum: number;
   mandatoryLeft: number;
}

/**
 * подсчёт необходимого числа апрувов по категориям
 */
export function getApprovalsCount({
   ticket,
   approvalPermissions,
   approvalPolicySpec,
}: {
   ticket: DeployTicket;
   approvalPermissions: ApprovalPermissionsResult;
   approvalPolicySpec: ApprovalPolicySpec | null;
}): ApprovalsCount {
   const approvalsCount = approvalPolicySpec?.approvalsCount || 0;
   const { approval } = ticket;
   const { mandatory: mandatoryUserApprovals } = getMandatoryApprovals({
      ticket,
      approvalPermissions,
   });
   const approvalsSumCount = approval.approve.size;
   const mandatoryApprovalsSumCount = [...mandatoryUserApprovals.values()].reduce(
      (sum, login) => sum + (approval.approve.has(login) ? 1 : 0),
      0,
   );
   let approvalsLeftCount = Math.max(approvalsCount - approvalsSumCount, 0);
   const mandatoryApprovalsLeftCount = Math.max(1 - mandatoryApprovalsSumCount, 0);
   if (approvalsSumCount >= approvalsCount && isMandatory(approvalPolicySpec) && mandatoryApprovalsLeftCount > 0) {
      // когда остался только mandatory
      approvalsLeftCount = 1;
   }

   return {
      needed: approvalsCount,
      sum: approvalsSumCount,
      mandatorySum: mandatoryApprovalsSumCount,
      left: approvalsLeftCount,
      mandatoryLeft: mandatoryApprovalsLeftCount,
   };
}

interface MandatoryApprovals {
   mandatory: Set<string>;
}

/**
 * полученные апрувы от пользователей, которые имеют mandatory роль
 */
export function getMandatoryApprovals({
   ticket,
   approvalPermissions,
}: {
   ticket: DeployTicket;
   approvalPermissions: ApprovalPermissionsResult;
}): MandatoryApprovals {
   const { approval, stageId } = ticket;
   const mandatory = new Set(
      [...(approval?.data?.keys() ?? [])].filter(
         login => approvalPermissions[createKey({ login, stageId, type: 'mandatory' })],
      ),
   );

   return {
      mandatory,
   };
}

export function isMandatory(approvalPolicySpec: ApprovalPolicySpec | null): boolean {
   return approvalPolicySpec?.policy === ApprovalPolicyType.Mandatory;
}

export function isWaitingForApprove(ticket: DeployTicket, actionsAvailable: Set<TicketAction>): boolean {
   return ticket.status === TicketStatus.WaitingForCommit && !actionsAvailable.has(TicketAction.Commit);
}

export function isWaitingForCommit(ticket: DeployTicket, actionsAvailable: Set<TicketAction>): boolean {
   return ticket.status === TicketStatus.WaitingForCommit && actionsAvailable.has(TicketAction.Commit);
}

/**
 * Когда изменяем политику апрувов, статус у тикета не меняется, так как статус меняется только при действиях.
 * Поэтому при снижении количества необходимых апрувов статус у тикета не поменяется
 */
export function isApprovalChanged(ticket: DeployTicket, approvalsCount: ApprovalsCount): boolean {
   const approvalStatus = ticket.approval?.status || '';
   return approvalStatus === ApprovalStatus.NOT_APPROVED && approvalsCount.left === 0;
}
