import { TDeployTicketSpec_EDeployTicketSourceType } from '../../../../proto-typings';
import { DeepReadonly } from '../../../typeHelpers';
import { TicketAction } from './actions';
import { isWaitingForApprove, isWaitingForCommit } from './approval';
import { DeployTicket } from './DeployTicket';
import { ActiveTicketStatuses, OnlyTicketStatuses, TicketStatus } from './statuses';

export const statusToQueryMap: { [key in TicketStatus]?: string } = {
   [TicketStatus.WaitingForCommit]:
      '([/status/action]=NULL OR [/status/action/type]=NULL OR [/status/action/type]="none")',
   [TicketStatus.Committed]: '([/status/action/type]="commit" AND [/status/progress]=NULL)',
   [TicketStatus.Pending]: '([/status/action/type]="commit" AND [/status/progress/pending/status]="true")',
   [TicketStatus.InProgress]: '([/status/action/type]="commit" AND [/status/progress/in_progress/status]="true")',
   [TicketStatus.OnHold]: '([/status/action/type]="on_hold" AND (NOT [/status/progress/closed/status]="true"))',
   [TicketStatus.Wait]: '([/status/action/type]="wait" AND (NOT [/status/progress/closed/status]="true"))',
   [TicketStatus.Closed]: '[/status/progress/closed/status]="true"',
   [TicketStatus.Skip]: '[/status/action/type]="skip"',
};
// может быть слишком тяжелым запросом
statusToQueryMap[TicketStatus.Unknown] = `NOT (${Object.values(statusToQueryMap).join(' OR ')})`;
// WaitingForApprove будет вычислен на UI на основании WaitingForCommit тикетах и данных об Approval,
// но не будем дублировать одинаковый запрос для WaitingForApprove и WaitingForCommit
statusToQueryMap[TicketStatus.WaitingForApprove] = statusToQueryMap[TicketStatus.WaitingForCommit];

export const draftLabel = 'draft:all';
export type RuleLabelTemplate = `rule:${string}`;
export type ReleaseRuleIdsAndDrafts = Set<typeof draftLabel | RuleLabelTemplate>;

export interface DeployTicketFilters {
   ticketId?: string;
   releaseId?: string;
   stageId?: string;
   releaseRuleIdsAndDrafts: ReleaseRuleIdsAndDrafts;
   ticketStatuses: Set<TicketStatus>;
}

// фильтры, выставляемые для удобства
export const initialDeployTicketFilters: DeployTicketFilters = {
   releaseRuleIdsAndDrafts: new Set(),
   ticketStatuses: new Set(ActiveTicketStatuses),
   ticketId: '',
   releaseId: '',
   stageId: '',
};

// пустые фильтры
export const emptyDeployTicketFilters: DeployTicketFilters = {
   releaseRuleIdsAndDrafts: new Set(),
   ticketStatuses: new Set(),
};

export function filterDeployTickets(filters: DeployTicketFilters): (ticket: DeployTicket) => boolean {
   const { ticketId, stageId, releaseId, releaseRuleIdsAndDrafts, ticketStatuses } = filters;
   return ticket => {
      const { sourceType, releaseRuleId, releaseId: ticketReleaseId, id, stageId: ticketStageId, status } = ticket;
      const isReleaseTicket = sourceType === TDeployTicketSpec_EDeployTicketSourceType.RELEASE_INTEGRATION;
      const isDraftTicket = sourceType === TDeployTicketSpec_EDeployTicketSourceType.STAGE_DRAFT;
      if (ticketId && !id.includes(ticketId)) {
         return false;
      }
      if (stageId && !ticketStageId.includes(stageId)) {
         return false;
      }
      if (releaseId && (isDraftTicket || !ticketReleaseId.includes(releaseId))) {
         return false;
      }
      if (releaseRuleIdsAndDrafts.size > 0) {
         if (isDraftTicket && !releaseRuleIdsAndDrafts.has('draft:all')) {
            return false;
         }
         if (isReleaseTicket && !releaseRuleIdsAndDrafts.has(`rule:${releaseRuleId}` as const)) {
            return false;
         }
      }
      if (ticketStatuses.size > 0 && !ticketStatuses.has(status)) {
         return false;
      }
      return true;
   };
}

/**
 * фильтрация на месте на основе информации об апрувах
 */
export function filterDeployTicketsByApprovalStatuses(
   approveFilter: boolean,
   commitFilter: boolean,
   getActionAvailable: (ticket: DeployTicket) => Set<TicketAction>,
): (ticket: DeployTicket) => boolean {
   return ticket => {
      if (!(approveFilter && commitFilter)) {
         const actionsAvailable = getActionAvailable(ticket);
         if (!approveFilter && isWaitingForApprove(ticket, actionsAvailable)) {
            return false;
         }
         if (!commitFilter && isWaitingForCommit(ticket, actionsAvailable)) {
            return false;
         }
      }

      return true;
   };
}

export const prepareFilterQuery = (filters: DeepReadonly<DeployTicketFilters>): string[] => {
   const filterQueryParams = [];
   const { ticketId, stageId, releaseId, releaseRuleIdsAndDrafts, ticketStatuses } = filters;

   if (ticketId) {
      filterQueryParams.push(`is_substr(lower("${ticketId}"), lower([/meta/id]))`);
   }
   if (releaseId) {
      filterQueryParams.push(
         `([/spec/source_type]!="stage_draft" AND is_substr(lower("${releaseId}"), lower([/spec/release_id])))`,
      );
   }
   if (stageId) {
      filterQueryParams.push(`is_substr(lower("${stageId}"), lower([/meta/stage_id]))`);
   }
   if (releaseRuleIdsAndDrafts.size > 0) {
      const rulesFilters = [];

      if (releaseRuleIdsAndDrafts.has('draft:all')) {
         rulesFilters.push(`[/spec/source_type]="stage_draft"`);
      }

      const releaseRules = [...releaseRuleIdsAndDrafts.values()]
         .filter(i => i.startsWith('rule:'))
         .map(i => i.slice(5));
      if (releaseRules.length > 0) {
         rulesFilters.push(`[/spec/release_rule_id] IN (${releaseRules.map(i => `"${i}"`).join(', ')})`);
      }

      if (rulesFilters.length > 0) {
         filterQueryParams.push(`(${rulesFilters.join(' OR ')})`);
      }
   }
   if (ticketStatuses.size > 0 && ticketStatuses.size !== OnlyTicketStatuses.size) {
      const statuses = new Set(ticketStatuses);
      if (statuses.has(TicketStatus.WaitingForApprove) && statuses.has(TicketStatus.WaitingForCommit)) {
         // на бекенде нет статуса апрува
         statuses.delete(TicketStatus.WaitingForApprove);
      }

      const statusesFilters = [...statuses.values()]
         .map(status => statusToQueryMap[status])
         .filter(statusQuery => statusQuery !== undefined);

      if (statusesFilters.length > 0) {
         filterQueryParams.push(`(${statusesFilters.join(' OR ')})`);
      }
   }
   return filterQueryParams;
};
