import { IJsonable } from '@yandex-infracloud-ui/libs';

import { EConditionStatus, TDockerRelease, TRelease, TSandboxRelease } from '../../../../proto-typings';
import { Entity } from '../../../../redux/models';
import { notEmpty } from '../../../../utils';
import { DeepPartial } from '../../../typeHelpers';

import { ReleaseRuleType } from '../releaseRule/ReleaseRuleFormParams';
import { ReleaseObjectStatus } from './statuses';

export interface Release {
   author: string;
   description: string;
   title: string;
   docker: DeepPartial<TDockerRelease> | null;
   id: string;
   sandbox: DeepPartial<TSandboxRelease> | null;
   type: ReleaseRuleType;
   creationTime: number;
   status: ReleaseObjectStatus;
}

/**
 * Эта мета-сущность.
 * Представляет собой релиз Sandbox или релиз Docker
 *
 * Создается вне рамок деплоя, именно факт создания релиза вызывает создания тикета.
 *
 * Как следствие - ничего не знает об объектах Deploy.
 */
export class ReleaseConverter implements Entity, Release, IJsonable {
   public static fromApi(raw: TRelease): Release {
      return new ReleaseConverter(raw).toJSON();
   }

   public author: string;

   public title: string;

   public description: string;

   public docker: DeepPartial<TDockerRelease> | null = null;

   public id: string;

   public sandbox: DeepPartial<TSandboxRelease> | null = null;

   public type: ReleaseRuleType;

   public creationTime: number;

   public status: ReleaseObjectStatus;

   constructor(private raw: TRelease) {
      this.id = this.raw.meta!.id!;
      this.type = this.getType();
      this.author = this.getAuthor();
      this.title = this.getTitle();
      this.description = this.getDescription();

      switch (this.type) {
         case ReleaseRuleType.Sandbox: {
            this.sandbox = this.raw.spec?.sandbox ?? null;
            break;
         }

         case ReleaseRuleType.Docker: {
            this.docker = this.raw.spec?.docker ?? null;
            break;
         }
      }

      this.creationTime = this.raw.meta?.creation_time ?? 0;
      this.status = this.getStatus();
   }

   public toJSON(): Release {
      return {
         id: this.id,
         type: this.type,
         author: this.author,
         sandbox: this.sandbox,
         docker: this.docker,
         description: this.description,
         title: this.title,
         creationTime: this.creationTime,
         status: this.status,
      };
   }

   private getAuthor(): string {
      switch (this.type) {
         case ReleaseRuleType.Docker:
            return this.raw.spec!.docker!.release_author;

         case ReleaseRuleType.Sandbox:
            return this.raw.spec!.sandbox!.release_author;

         default:
            console.warn('Unknown release type', this.type);
            return '';
      }
   }

   private getDescription(): string {
      const description = this.raw.spec?.description || '';

      switch (this.type) {
         case ReleaseRuleType.Docker:
            return description;

         case ReleaseRuleType.Sandbox: {
            const subDescription = this.sandbox?.description || '';

            return [description, subDescription].filter(notEmpty).join(', ');
         }

         default: {
            console.warn('Unknown release type', this.type);
            return '';
         }
      }
   }

   private getTitle(): string {
      const title = this.raw.spec?.title || '';

      switch (this.type) {
         case ReleaseRuleType.Docker:
            return title;
         case ReleaseRuleType.Sandbox: {
            const subtitle = this.sandbox?.title || '';

            return [title, subtitle].filter(notEmpty).join(', ');
         }

         default:
            console.warn('Unknown release type', this.type);
            return '';
      }
   }

   private getType(): ReleaseRuleType {
      return this.raw.spec?.sandbox ? ReleaseRuleType.Sandbox : ReleaseRuleType.Docker;
   }

   private getStatus(): ReleaseObjectStatus {
      const releaseStatus = this.raw.status!;
      const { processing, progress } = releaseStatus;
      let status: ReleaseObjectStatus = ReleaseObjectStatus.Created;
      if (processing?.finished?.status === EConditionStatus.CS_TRUE) {
         status = ReleaseObjectStatus.Open;
      }
      if (!progress) {
         return status;
      }

      const map = [
         [ReleaseObjectStatus.Pending, progress.pending],
         [ReleaseObjectStatus.InProgress, progress.in_progress],
         [ReleaseObjectStatus.Closed, progress.closed],
      ] as const;

      for (const [currentStatus, condition] of map) {
         if (condition?.status === EConditionStatus.CS_TRUE) {
            return currentStatus;
         }
      }

      return ReleaseObjectStatus.Unknown;
   }
}
