import { IJsonable } from '@yandex-infracloud-ui/libs';

import { TAutoCommitPolicy_EType, TReleaseRule } from '../../../../proto-typings';
import { ResourceType } from '../../stage/DeployUnit';

export enum ReleaseRuleType {
   Sandbox = 'sandbox',
   Docker = 'docker',
}

export const DefaultReleaseRuleType = ReleaseRuleType.Sandbox;

export enum ReleaseType {
   Stable = 'stable',
   Prestable = 'prestable',
   Testing = 'testing',
   Unstable = 'unstable',
}

export interface DockerPatch {
   boxId: string;
   deployUnitId: string;
   id: string;
}

export interface SandboxPatch {
   deployUnitId: string;
   id: string;
   ref: string;
   resourceType: string;
   type: ResourceType;
   deployGroupMark?: string;
}

export interface SandboxAttribute {
   key: string;
   value: string;
}

export interface SandboxFormParams {
   attributes: SandboxAttribute[];
   patches: SandboxPatch[];
   resources: string[];
   taskType: string;
}

export interface DockerFormParams {
   image: string;
   patches: DockerPatch[];
}

export interface ReleaseRuleFormParams {
   autocommit: boolean;
   description: string;
   docker?: DockerFormParams; // docker
   id: string;
   releaseTypes: (ReleaseType | string)[];
   sandbox?: SandboxFormParams; // sandbox
   type: ReleaseRuleType; // sandbox | docker
}

export const defaultDockerPatch: DockerPatch = {
   boxId: '',
   deployUnitId: '',
   id: '',
};

export const defaultSandboxPatch: SandboxPatch = {
   id: '',
   resourceType: '',
   ref: '',
   deployUnitId: '',
   type: ResourceType.Layer,
};

export const releaseRuleInitialFormParams: ReleaseRuleFormParams = {
   autocommit: false,
   description: '',
   docker: {
      image: '',
      patches: [defaultDockerPatch],
   },
   id: '',
   releaseTypes: [],
   sandbox: {
      attributes: [],
      taskType: '',
      resources: [],
      patches: [defaultSandboxPatch],
   },
   type: DefaultReleaseRuleType,
};

export class ReleaseRuleFormParamsConverter implements ReleaseRuleFormParams, IJsonable {
   public static fromApi(raw: TReleaseRule): ReleaseRuleFormParams {
      return new ReleaseRuleFormParamsConverter(raw).toJSON();
   }

   public readonly autocommit: boolean;

   public readonly description: string;

   public readonly docker: DockerFormParams | undefined;

   public readonly id: string;

   public readonly releaseTypes: (ReleaseType | string)[] = [];

   public readonly sandbox: SandboxFormParams | undefined;

   public readonly type: ReleaseRuleType;

   constructor(public raw: TReleaseRule) {
      this.id = this.raw.meta!.id;
      this.description = this.raw.spec?.description ?? '';
      this.type = this.getType();
      this.autocommit = this.getAutoCommit();

      switch (this.type) {
         case ReleaseRuleType.Sandbox: {
            this.releaseTypes = (this.raw.spec?.sandbox?.release_types as (ReleaseType | string)[]) ?? [];
            this.sandbox = this.getSandboxParams();
            break;
         }

         case ReleaseRuleType.Docker: {
            this.releaseTypes = (this.raw.spec?.docker?.release_types as (ReleaseType | string)[]) ?? [];
            this.docker = this.getDockerParams();
            break;
         }
      }
   }

   public toJSON(): ReleaseRuleFormParams {
      return {
         autocommit: this.autocommit,
         description: this.description,
         docker: this.docker,
         id: this.id,
         releaseTypes: this.releaseTypes,
         sandbox: this.sandbox,
         type: this.type,
      };
   }

   private getAutoCommit() {
      const type = this.raw.spec?.auto_commit_policy?.type ?? TAutoCommitPolicy_EType.NONE;

      return type !== TAutoCommitPolicy_EType.NONE;
   }

   private getDockerParams(): DockerFormParams {
      return {
         image: this.raw.spec!.docker!.image_name,
         patches: Object.entries(this.raw.spec!.patches).map(
            ([patchId, { docker }]) =>
               ({
                  id: patchId,
                  boxId: docker?.docker_image_ref?.box_id,
                  deployUnitId: docker?.docker_image_ref?.deploy_unit_id,
               } as DockerPatch),
         ),
      };
   }

   private getSandboxParams(): SandboxFormParams {
      const config = this.raw.spec!.sandbox!;

      return {
         attributes: Object.keys(config.attributes ?? {}).map(key => ({ key, value: config.attributes[key] ?? '' })),
         patches: Object.entries(this.raw.spec!.patches).map(([patchId, { sandbox }]) => {
            const type = sandbox?.dynamic?.dynamic_resource_id
               ? ResourceType.DynamicResource
               : sandbox?.static?.layer_ref
               ? ResourceType.Layer
               : ResourceType.StaticResource;

            if (type === ResourceType.DynamicResource) {
               return {
                  deployGroupMark: sandbox?.dynamic?.deploy_group_mark,
                  id: patchId,
                  ref: sandbox?.dynamic?.dynamic_resource_id,
                  resourceType: sandbox?.sandbox_resource_type,
                  type,
               } as SandboxPatch;
            }

            return {
               deployUnitId: sandbox?.static?.deploy_unit_id,
               id: patchId,
               ref: type === ResourceType.Layer ? sandbox?.static?.layer_ref : sandbox?.static?.static_resource_ref,
               resourceType: sandbox?.sandbox_resource_type,
               type,
            } as SandboxPatch;
         }),
         resources: config.resource_types,
         taskType: config.task_type,
      };
   }

   private getType(): ReleaseRuleType {
      return this.raw.spec?.sandbox ? ReleaseRuleType.Sandbox : ReleaseRuleType.Docker;
   }
}
