import { TPod, TPodStatus_TScheduling } from '../../../../proto-typings';
import { getNormalizeList } from '../../list';
import { getLevelWarningCount, getMapWarningCount, getPodWarning } from '../errorConvertor';
import { getStatusStateInfo } from '../status';
import { Pod, PodBox, PodLayer, PodStaticResource, PodVolume, PodWorkload, SchedulingInfo, PodObject } from './model';
import { getPodBox } from './PodBoxConvertor';
import { getPodLayer } from './PodLayerConvertor';
import { getPodStatisResource } from './PodStaticResourceConvertor';
import { getPodVolume } from './PodVolumeConvertor';
import { getPodWorkload } from './PodWorkloadConvertor';

const getWorkloadListData = getNormalizeList({
   selectId: (workload: PodWorkload) => workload.id,
   groupBy: {
      box: workload => workload.boxId ?? '',
   },
});

const getBoxListData = getNormalizeList({
   selectId: (box: PodBox) => box.id,
});

const getLayerListData = getNormalizeList({
   selectId: (layer: PodLayer) => layer.id,
});

const getStaticResourceListData = getNormalizeList({
   selectId: (resource: PodStaticResource) => resource.id,
});

const getVolumeListData = getNormalizeList({
   selectId: (volume: PodVolume) => volume.id,
});

const addTargetRevision = (revision: number | null) => <T extends PodObject>(map: ReadonlyMap<string, T>) => {
   if (revision === null) {
      return map;
   }
   for (const object of map.values()) {
      object.targetRevision = revision;
   }
   return map;
};

function getPod(rawPod: TPod): Pod {
   const { spec, status, meta } = rawPod;

   const { id } = meta ?? {};

   const { pod_agent_payload: podAgentPayloadSpec, node_id } = spec ?? {};
   const { spec: agentSpec } = podAgentPayloadSpec ?? {};
   const { revision: rawTargetRevision, workloads: workloadSpecs } = agentSpec ?? {};
   const targetRevision = rawTargetRevision ?? null;

   // получение вложенных полей
   const { agent, scheduling, dns } = status ?? {};
   const { pod_agent_payload, execution_error, install_error, state, validation_failures, last_heartbeat_time } =
      agent ?? {};
   const { status: podAgentStatus } = pod_agent_payload ?? {};
   const { persistent_fqdn } = dns ?? {};

   const { revision, workloads: workloadStatuses, boxes: boxStatuses, volumes, resource_gang } = podAgentStatus ?? {};
   const { layers, static_resources } = resource_gang ?? {};

   // информация о списках и связях между сущностями
   const workloadSpecsMap = new Map(workloadSpecs?.map(workload => [workload.id, workload]));
   const { values: workloadsValues, groupBy: workloadsGroupBy } = getWorkloadListData({
      list: workloadStatuses?.map(rawWorkloadStatus =>
         getPodWorkload(rawWorkloadStatus, workloadSpecsMap.get(rawWorkloadStatus.id) ?? null),
      ),
   });

   const { values: boxValues, idList: boxIdList } = getBoxListData({ list: boxStatuses?.map(getPodBox) });

   const layerMap: Pod['layerMap'] = new Map(boxStatuses?.map(box => [box.id, new Set(box.rootfs_layer_refs)]));
   const staticResourceMap: Pod['staticResourceMap'] = new Map(
      boxStatuses?.map(box => [box.id, new Set(box.static_resource_refs)]),
   );

   // что делать с ворклоадами, которые привязаны к несуществующим боксам или вообще не имеют привязки?
   const workloadMap: Map<string, Set<string>> = new Map(
      boxIdList.map(boxId => [boxId, new Set(Array.from(workloadsGroupBy.box.get(boxId) ?? []).map(w => w.id))]),
   );

   const { values: layersValues } = getLayerListData({ list: layers?.map(getPodLayer) });
   const { values: staticResourceValues } = getStaticResourceListData({
      list: static_resources?.map(getPodStatisResource),
   });
   const { values: volumeValues } = getVolumeListData({ list: volumes?.map(getPodVolume) });

   // приписывание ревизии
   // может сделать нормально с зависимостями?
   const withTargetRevision = addTargetRevision(targetRevision);

   const podStatus: Pod = {
      id: id ?? '',
      nodeId: node_id ?? null,
      currentState: state ?? null,
      revision: revision ?? null,
      targetRevision,

      statusStateInfo: getStatusStateInfo(podAgentStatus ?? null),
      schedulingInfo: getSchedulingInfo(scheduling ?? null),

      workloads: withTargetRevision(workloadsValues),
      layers: withTargetRevision(layersValues),
      staticResources: withTargetRevision(staticResourceValues),
      volumes: withTargetRevision(volumeValues),
      boxes: withTargetRevision(boxValues),

      workloadMap,
      layerMap,
      staticResourceMap,
      errors: {
         install: install_error ?? null,
         execution: execution_error ?? null,
      },
      validationErrors: validation_failures ?? [],
      persistentFqdn: persistent_fqdn ?? null,
      warning: {},
      warningCount: 0,
      totalWarningCount: 0,
      agentLastHeartbeatTime: last_heartbeat_time ?? null,
   };

   podStatus.warning = getPodWarning(podStatus);
   podStatus.warningCount = getLevelWarningCount(podStatus.warning);
   podStatus.totalWarningCount =
      podStatus.warningCount +
      getMapWarningCount(podStatus.workloads) +
      getMapWarningCount(podStatus.boxes) +
      getMapWarningCount(podStatus.layers) +
      getMapWarningCount(podStatus.volumes) +
      getMapWarningCount(podStatus.staticResources);

   return podStatus;
}

function getSchedulingInfo(scheduling: TPodStatus_TScheduling | null): SchedulingInfo {
   const { state, error } = scheduling ?? {};
   return {
      state: state ?? null,
      error: error ?? null,
   };
}

export const PodConverter = {
   getPod,
   /**
    * фиксированный способ получить id подсета по соглашениям деплоя
    */
   getPodSetId: (stageId: string, duId: string) => `${stageId}.${duId}`,
};
