import {
   EBoxState,
   ELayerState,
   EPodCurrentState,
   ESchedulingState,
   EStaticResourceState,
   EVolumeState,
   EWorkloadState,
   TDownloadProgress,
   TError,
} from '../../../../proto-typings';
import { FailStatus, WarningLevelData } from '../errors';
import { StatusStateInfo } from '../status';

type BoxId = string;
type WorkloadId = string;
type LayerId = string;
type ResourceId = string;
type VolumeId = string;

export interface SchedulingInfo {
   state: ESchedulingState | null;
   error: TError | null;
}

export enum PodObjectName {
   Box,
   Workload,
   Layer,
   StaticResource,
   Volume,
}

export type PodObjectType = {
   [PodObjectName.Box]: PodBox;
   [PodObjectName.Workload]: PodWorkload;
   [PodObjectName.Layer]: PodLayer;
   [PodObjectName.StaticResource]: PodStaticResource;
   [PodObjectName.Volume]: PodVolume;
};

export const PodObjectTitle: Record<PodObjectName, string> = {
   [PodObjectName.Box]: 'Box',
   [PodObjectName.Workload]: 'Workload',
   [PodObjectName.Layer]: 'Layer',
   [PodObjectName.StaticResource]: 'Static Resource',
   [PodObjectName.Volume]: 'Volume',
};

export interface PodObject {
   id: string;
   statusStateInfo: StatusStateInfo;
   revision: number | null;
   targetRevision: number | null;
   warning: WarningLevelData<string>;
   warningCount: number;
}

export interface PodWorkload extends PodObject {
   boxId: string | null;
   state: EWorkloadState;
   startStatus: FailStatus;
   initStatuses: FailStatus[];
   readinessStatus: FailStatus;
   livenessStatus: FailStatus;
   stopStatus: FailStatus;
   destroyStatus: FailStatus;
}

export interface PodBox extends PodObject {
   state: EBoxState;
   initStatuses: FailStatus[];
   ip6Address: string | null;
   type: string;
   isSystemBox: boolean;
}

export interface PodLayer extends PodObject {
   state: ELayerState;
   downloadProgress: TDownloadProgress | null;
}

export interface PodStaticResource extends PodObject {
   state: EStaticResourceState;
   downloadProgress: TDownloadProgress | null;
}

export interface PodVolume extends PodObject {
   state: EVolumeState;
}

interface PodErrors {
   install: TError | null;
   execution: TError | null;
}

export enum PodCustomCurrentState {
   PCS_SUSPECTED = 'suspected',
}

export type PodCurrentState = EPodCurrentState | PodCustomCurrentState;

export interface Pod extends PodObject {
   nodeId: string | null;
   currentState: PodCurrentState | null;

   // вложенные сущности
   workloads: ReadonlyMap<WorkloadId, PodWorkload>;
   boxes: ReadonlyMap<BoxId, PodBox>;
   layers: ReadonlyMap<LayerId, PodLayer>;
   volumes: ReadonlyMap<VolumeId, PodVolume>;
   staticResources: ReadonlyMap<ResourceId, PodStaticResource>;

   // связи между сущностями
   workloadMap: ReadonlyMap<BoxId, Set<WorkloadId>>;
   layerMap: ReadonlyMap<BoxId, Set<LayerId>>;
   staticResourceMap: ReadonlyMap<BoxId, Set<ResourceId>>;

   errors: PodErrors;
   schedulingInfo: SchedulingInfo;
   validationErrors: string[];
   persistentFqdn: string | null;

   totalWarningCount: number;
   agentLastHeartbeatTime: number | null;
}
