import {
   EConditionStatus,
   EObjectType,
   TDeployStatusDetails,
   TMultiClusterReplicaSet,
   TMultiClusterReplicaSetSpec,
   TReplicaSet,
} from '../../../../proto-typings';
import { decodeYPErrorMessage } from '../../../../utils';
import { YpLocation } from '../../../api';
import { getStatusStateInfo, InfoStatus, StatusStateInfo } from '../status';
import {
   getEmptyReplicaSetPodsSummary,
   ReplicaSet,
   ReplicaSetLabels,
   ReplicaSetPodsSummary,
   ReplicaSetType,
} from './model';

function getNumberFromString(n?: string): number | null {
   if (!n) {
      return null;
   }
   const revisionIdNumber = Number(n);
   if (!isNaN(revisionIdNumber)) {
      return revisionIdNumber;
   }
   return null;
}

function getReplicaSetType(rawType: EObjectType): ReplicaSetType {
   if (rawType === EObjectType.OT_MULTI_CLUSTER_REPLICA_SET) {
      return ReplicaSetType.MultiCluster;
   }
   return ReplicaSetType.PerCluster;
}

function getMCRSLocations(rawSpec: TMultiClusterReplicaSetSpec): Set<string> {
   const { clusters } = rawSpec;

   return new Set(clusters?.map(item => item.cluster));
}

function getReplicaSetPodsSummary(
   mappedRevisions: TDeployStatusDetails['mapped_revisions'],
   currentRevision: number,
   specTotalPods: number,
   podsUnavailable: number,
): ReplicaSetPodsSummary {
   const podsSummary: ReplicaSetPodsSummary = getEmptyReplicaSetPodsSummary();

   podsSummary.total += podsUnavailable;
   podsSummary.unavailable = podsUnavailable;

   Object.keys(mappedRevisions).forEach(stringKey => {
      const value = mappedRevisions[stringKey];
      const { pods_total = 0, pods_in_progress = 0, pods_failed = 0, pods_ready = 0 } = value ?? {};

      podsSummary.total += pods_total;

      const key = Number(stringKey);
      if (key === currentRevision) {
         podsSummary.ready += pods_ready;
         podsSummary.inProgress += pods_in_progress;
         podsSummary.failed += pods_failed;
      } else if (key < currentRevision) {
         podsSummary.oldReady += pods_ready;
         podsSummary.oldInProgress += pods_in_progress;
         podsSummary.oldFailed += pods_failed;
      }
   });

   // если по какой-то причине в спеке больше подов, чем в статусе,
   // то разницу пишем в unknown, а total берём из спеки
   if (specTotalPods > podsSummary.total) {
      podsSummary.unknown = specTotalPods - podsSummary.total;
      podsSummary.total = specTotalPods;
   }

   return podsSummary;
}

function getLastAttemptMessage(deployStatusDetails?: TDeployStatusDetails): string | null {
   const controllerStatus = deployStatusDetails?.controller_status;
   const lastAttempt = controllerStatus?.last_attempt;
   const succeeded = lastAttempt?.succeeded;

   if (succeeded && succeeded.status === EConditionStatus.CS_FALSE && succeeded.message) {
      const lastAttemptMessage = decodeYPErrorMessage(succeeded.message);
      return lastAttemptMessage;
   }
   return null;
}

/** Единый вид replica set с учётом типа */
export function getReplicaSetUIModel<T extends TReplicaSet | TMultiClusterReplicaSet>(
   rawReplicaSet: T,
   location?: string,
): ReplicaSet {
   const ypLocation = location ?? YpLocation.XDC;
   const { meta, labels: rawLabels } = rawReplicaSet;

   const { id, type: rawType } = meta!;

   const rawPCRS = rawReplicaSet as TReplicaSet;

   const rawMCRS = rawReplicaSet as TMultiClusterReplicaSet;

   const type = getReplicaSetType(rawType);

   if (type === ReplicaSetType.PerCluster && !location) {
      throw new Error('Need location for PerCluster replica set');
   }

   const locations = type === ReplicaSetType.PerCluster ? new Set([location!]) : getMCRSLocations(rawMCRS.spec!);

   let currentRevision: number | null = null;
   let targetRevision: number | null = null;
   if (type === ReplicaSetType.PerCluster) {
      currentRevision = getNumberFromString(rawPCRS.status?.revision_id);
      targetRevision = getNumberFromString(rawPCRS.spec?.revision_id);
   } else {
      currentRevision = rawMCRS.status?.revision ?? null;
      targetRevision = rawMCRS.spec?.revision ?? null;
   }

   const deployStatusDetails =
      type === ReplicaSetType.PerCluster
         ? rawPCRS.status?.deploy_status?.details
         : rawMCRS.status?.multi_cluster_deploy_status?.details;

   // нужно брать из спеки RS/MCRS #DEPLOY-4823
   const specTotalPods =
      type === ReplicaSetType.PerCluster
         ? rawPCRS.spec?.replica_count
         : rawMCRS.spec?.clusters.map(v => v.spec?.replica_count ?? 0).reduce((acc, value) => acc + value, 0);

   // у RS нет такого поля
   // @reddi: в rsc, если есть недоступные поды, то он не запускается, так как там кластер один и он считает его неживым
   const podsUnavailable =
      type === ReplicaSetType.PerCluster ? 0 : rawMCRS.status?.multi_cluster_deploy_status?.pods_unavailable;

   // расхождение типов с апи
   const podsSummary = getReplicaSetPodsSummary(
      deployStatusDetails?.mapped_revisions ?? {},
      currentRevision ?? 0, // ревизия не может быть отрицательной
      specTotalPods ?? 0,
      podsUnavailable ?? 0,
   );

   const lastAttemptMessages: Map<string, string> = new Map();
   const podsSummaryByLocation: Record<string, ReplicaSetPodsSummary> = {};

   if (type === ReplicaSetType.PerCluster) {
      podsSummaryByLocation[location!] = podsSummary;
   } else {
      const statusesByLocation = rawMCRS.status?.mapped_cluster_deploy_statuses ?? {};
      for (const replicaSetLocation of locations.values()) {
         const status = statusesByLocation[replicaSetLocation];
         const specTotalPodsByLocation = rawMCRS.spec?.clusters?.find(v => v.cluster === replicaSetLocation)?.spec
            ?.replica_count;

         if (status) {
            podsSummaryByLocation[replicaSetLocation] = getReplicaSetPodsSummary(
               status.details?.mapped_revisions ?? {},
               currentRevision ?? 0, // ревизия не может быть отрицательной
               specTotalPodsByLocation ?? 0,
               0, // podsUnavailable только для MCRS, не считаем по кластерам
            );
            const lastAttemptMessage = getLastAttemptMessage(status.details);
            if (lastAttemptMessage) {
               lastAttemptMessages.set(replicaSetLocation, lastAttemptMessage);
            }
         }
      }
   }

   const lastAttemptMessage = getLastAttemptMessage(deployStatusDetails);
   if (lastAttemptMessage) {
      lastAttemptMessages.set(ypLocation, lastAttemptMessage);
   }

   const labels: ReplicaSetLabels = {
      'yd.deploy_speed': rawLabels?.['yd.deploy_speed'] ?? null,
   };

   const isPerCluster = type === ReplicaSetType.PerCluster;
   const rawInfo: InfoStatus = {
      ready: (isPerCluster ? rawPCRS.status?.ready_condition : rawMCRS.status?.ready) ?? null,
      in_progress: (isPerCluster ? rawPCRS.status?.in_progress_condition : rawMCRS.status?.in_progress) ?? null,
      failed: (isPerCluster ? rawPCRS.status?.failed_condition : rawMCRS.status?.failed) ?? null,
   };
   const statusStateInfo: StatusStateInfo = getStatusStateInfo(rawInfo);

   // cluster_succeeded ? — нет в прото

   return {
      id,
      type,
      locations,
      location: ypLocation,
      currentRevision,
      targetRevision,
      podsSummary,
      podsSummaryByLocation,
      lastAttemptMessages,
      statusStateInfo,
      labels,
   };
}
