import {
   EContainerState,
   TContainerStatus,
   TDestroyStatus,
   TError,
   THttpGetStatus,
   TLivenessStatus,
   TReadinessStatus,
   TStopStatus,
   TTcpCheckStatus,
   TUnixSignalStatus,
} from '../../../proto-typings';
import { createKey, parseYpDatetime, ProtoDate } from '../../../utils';
import { FailStatus, getDefaultFailStatus, WarningData, WarningLevelData } from './errors';
import { Pod, PodBox, PodLayer, PodObject, PodStaticResource, PodVolume, PodWorkload } from './Pod';
import { ReplicaSet } from './ReplicaSet';
import { StatusInfo } from './status';

export enum PodWarningLevel {
   Validation = 'pod validation',
   PodAgent = 'pod agent',
   NodeAgent = 'node agent',
   Install = 'pod install',
   Scheduling = 'pod scheduling',
}

export enum BoxWarningLevel {
   Box = 'box fail',
   Init = 'init command',
}
export type BoxWarning = WarningLevelData<BoxWarningLevel>;

export enum WorkloadWarningLevel {
   Workload = 'workload fail',
   Liveness = 'liveness probe',
   Readness = 'readiness probe',
   Stop = 'stop probe',
   Destroy = 'destroy probe',
   Init = 'init command',
   Start = 'start command',
}
export type WorkloadWarning = WarningLevelData<WorkloadWarningLevel>;

export enum LayerWarningLevel {
   Layer = 'layer fail',
}
export type LayerWarning = WarningLevelData<LayerWarningLevel>;

export enum StaticResourceWarningLevel {
   Resource = 'static resource fail',
}
export type StaticResourceWarning = WarningLevelData<StaticResourceWarningLevel>;

export enum VolumeWarningLevel {
   Volume = 'volume fail',
}
export type VolumeWarning = WarningLevelData<VolumeWarningLevel>;

export const getLevelWarningCount = (level: WarningLevelData<string>): number =>
   Object.values(level).reduce((sum, list) => {
      if (!list) {
         return sum;
      }
      return sum + list.length;
   }, 0);

export const getMapWarningCount = <T extends PodObject>(map: Map<string, T> | ReadonlyMap<string, T>): number =>
   [...map.values()].reduce((sum, level) => sum + level.warningCount, 0);

// TODO: схлопывать одинаковые ошибки
export type PodWarning = WarningLevelData<PodWarningLevel>;

export const replicaSetWarningLevels = ['replica set'] as const;
export type ReplicaSetWarningLevel = typeof replicaSetWarningLevels[number];

export interface ReplicaSetWarning {
   replicaSet: WarningLevelData<ReplicaSetWarningLevel>;
   count: number;
}

export interface Warning {
   level: string;
   clusters: Set<string>;
   sources: string[];
   counter: number;
}

type AddData = (...data: (WarningData | undefined | null)[]) => void;

const addWarningData = <T extends string>(levelData: WarningLevelData<T>) => (level: T): AddData => (...data) => {
   if (!levelData.hasOwnProperty(level)) {
      levelData[level] = [];
   }
   levelData[level]?.push(
      ...data.filter<WarningData>((e): e is WarningData => Boolean(e)),
   );
};

const addFailed = (add: AddData) => (failed: StatusInfo) => {
   if (failed.active) {
      add({
         failed,
      });
   }
};

const addError = (add: AddData) => (error: TError | null) => {
   if (error) {
      add({
         error,
      });
   }
};

const addFailStatus = (add: AddData) => (...statuses: (FailStatus | null | undefined)[]) => {
   add(...statuses.map(status => (status?.failReason ? { failStatus: status } : null)));
};

export function getPodWarning(pod: Pod): PodWarning {
   const podData: PodWarning = {};
   const addPodWarning = addWarningData(podData);

   addPodWarning(PodWarningLevel.Validation)(...pod.validationErrors.map(message => ({ message })));
   addFailed(addPodWarning(PodWarningLevel.PodAgent))(pod.statusStateInfo.failed);
   addError(addPodWarning(PodWarningLevel.NodeAgent))(pod.errors.execution);
   addError(addPodWarning(PodWarningLevel.Install))(pod.errors.install);
   addError(addPodWarning(PodWarningLevel.Scheduling))(pod.schedulingInfo.error);

   return podData;
}

export function getLayerWarning(layer: Omit<PodLayer, 'warning'>): PodLayer['warning'] {
   const layerData: PodLayer['warning'] = {};
   const addLayerWarning = addWarningData(layerData);

   addFailed(addLayerWarning(LayerWarningLevel.Layer))(layer.statusStateInfo.failed);

   return layerData;
}

export function getBoxWarning(box: Omit<PodBox, 'warning'>): PodBox['warning'] {
   const boxData: PodBox['warning'] = {};
   const addBoxWarning = addWarningData(boxData);

   addFailed(addBoxWarning(BoxWarningLevel.Box))(box.statusStateInfo.failed);
   addFailStatus(addBoxWarning(BoxWarningLevel.Init))(...box.initStatuses);

   return boxData;
}

export function getStaticResourceWarning(resource: Omit<PodStaticResource, 'warning'>): PodStaticResource['warning'] {
   const resourceData: PodStaticResource['warning'] = {};
   const addResourcesWarning = addWarningData(resourceData);

   addFailed(addResourcesWarning(StaticResourceWarningLevel.Resource))(resource.statusStateInfo.failed);

   return resourceData;
}

export function getVolumeWarning(volume: Omit<PodVolume, 'warning'>): PodVolume['warning'] {
   const volumeData: PodVolume['warning'] = {};
   const addVolumeWarning = addWarningData(volumeData);

   addFailed(addVolumeWarning(VolumeWarningLevel.Volume))(volume.statusStateInfo.failed);

   return volumeData;
}

export function getWorkloadWarning(workload: Omit<PodWorkload, 'warning'>): PodWorkload['warning'] {
   const workloadData: PodWorkload['warning'] = {};
   const addWorkloadWarning = addWarningData(workloadData);

   addFailed(addWorkloadWarning(WorkloadWarningLevel.Workload))(workload.statusStateInfo.failed);
   addFailStatus(addWorkloadWarning(WorkloadWarningLevel.Init))(...workload.initStatuses);
   addFailStatus(addWorkloadWarning(WorkloadWarningLevel.Start))(workload.startStatus);
   addFailStatus(addWorkloadWarning(WorkloadWarningLevel.Liveness))(workload.livenessStatus);
   addFailStatus(addWorkloadWarning(WorkloadWarningLevel.Readness))(workload.readinessStatus);
   addFailStatus(addWorkloadWarning(WorkloadWarningLevel.Stop))(workload.stopStatus);
   addFailStatus(addWorkloadWarning(WorkloadWarningLevel.Destroy))(workload.destroyStatus);

   return workloadData;
}

export function getReplicaSetWarning(replicaSet: ReplicaSet): ReplicaSetWarning {
   const { lastAttemptMessages } = replicaSet;
   const replicaSetData: ReplicaSetWarning['replicaSet'] = {};
   const addReplicaSetWarning = addWarningData(replicaSetData);
   const locationsByMessage: Map<string, Set<string>> = new Map(
      Array.from(lastAttemptMessages.values()).map(value => [value, new Set()]),
   );
   for (const [location, value] of lastAttemptMessages) {
      locationsByMessage.get(value)?.add(location);
   }

   addReplicaSetWarning('replica set')(
      ...Array.from(locationsByMessage).map(([message, locations]) => ({ message, meta: { locations } })),
   );

   const replicaSetWarning: ReplicaSetWarning = {
      replicaSet: replicaSetData,
      count: getLevelWarningCount(replicaSetData),
   };

   return replicaSetWarning;
}

// все ошибки для деплой юнита
interface DuWarningParams {
   pods: { cluster: string; pod: Pod }[];
   replicaSets: ReplicaSet[];
}

export interface DuWarningResult {
   replicaSets: Map<string, ReplicaSetWarning>;
   pods: Map<string, Pod>;
}

export const extractDeployUnitClusterErrors = ({ pods, replicaSets }: DuWarningParams): DuWarningResult => {
   const warnings: DuWarningResult = {
      replicaSets: new Map(
         replicaSets
            .map(
               replicaSet =>
                  [
                     createKey({ id: replicaSet.id, location: replicaSet.location }),
                     getReplicaSetWarning(replicaSet),
                  ] as const,
            )
            .filter(data => data[1].count > 0),
      ),
      pods: new Map(
         pods
            .filter(({ pod }) => pod.totalWarningCount > 0)
            .map(({ cluster, pod }) => [createKey({ id: pod.id, cluster }), pod] as const),
      ),
   };

   return warnings;
};

export const getDuWarningsCount = (warnings: DuWarningResult): number => {
   const { replicaSets, pods } = warnings;
   return (
      Array.from(replicaSets.values()).reduce((sum, data) => sum + data.count, 0) +
      Array.from(pods.values()).reduce((sum, data) => sum + data.totalWarningCount, 0)
   );
};

export function addTailsToFailStatus<T extends { stdout_tail?: unknown; stderr_tail?: unknown }>(
   rawStatus: T,
   status: FailStatus,
) {
   const { stderr_tail, stdout_tail, ...otherFields } = rawStatus;
   if (stderr_tail) {
      status.terminalTails.set('stderr', String(stderr_tail));
   }
   if (stdout_tail) {
      status.terminalTails.set('stdout', String(stdout_tail));
   }
   status.failReason = JSON.stringify(otherFields);
}

export const getContainerLastFailStatus = (status: TContainerStatus | null): FailStatus => {
   const result: FailStatus = getDefaultFailStatus();

   const { last } = status ?? {};
   if (!last) {
      // last === null => команда еще не запускалась, вообще не выводим ошибок
      return result;
   }
   const { state, return_code, start_time, death_time } = last;
   // валидность проверки??? при том, что есть поле last_failed
   // last.state === 'exited' && (last.return_code === undefined или last.return_code === 0) => success, тоже ничего не выводим
   const isOk = state === EContainerState.EContainerState_EXITED && (return_code === undefined || return_code === 0);
   if (!isOk) {
      addTailsToFailStatus(last, result);
      if (start_time) {
         result.times.set('start', Number(parseYpDatetime(start_time as ProtoDate)));
      }
      if (death_time) {
         result.times.set('death', Number(parseYpDatetime(death_time as ProtoDate)));
      }
   }

   return result;
};

export const getStartContainerLastFailStatus = (status: TContainerStatus | null): FailStatus => {
   const result: FailStatus = getDefaultFailStatus();

   const { last, current } = status ?? {};
   if (!last) {
      // last === null => команда еще не запускалась, вообще не выводим ошибок
      return result;
   }

   // current.state === 'running' => команда сейчас запущена и работает, тоже ничего не выводим
   if (!(current?.state === 'running')) {
      addTailsToFailStatus(last, result);
      const { start_time, death_time } = last;
      if (start_time) {
         result.times.set('start', Number(parseYpDatetime(start_time as ProtoDate)));
      }
      if (death_time) {
         result.times.set('death', Number(parseYpDatetime(death_time as ProtoDate)));
      }
   }

   return result;
};

const getTcpHttpUnixLastFailStatus = (
   status: TTcpCheckStatus | THttpGetStatus | TUnixSignalStatus | null,
): FailStatus => {
   const result: FailStatus = getDefaultFailStatus();
   if (!status) {
      return result;
   }

   // если last === null => команда еще не запускалась, вообще не выводим ошибок
   // last.state === 'success' => success, тоже ничего не выводим
   // если не success => выводим весь last объект в json
   const { last } = status;

   if (!last) {
      return result;
   }

   if (last.state === 'success') {
      return result;
   }

   result.failReason = JSON.stringify(last);
   if ('start_time' in last) {
      result.times.set('start', Number(parseYpDatetime(last.start_time as ProtoDate)));
   }
   if ('death_time' in last) {
      result.times.set('death', Number(parseYpDatetime(last.death_time as ProtoDate)));
   }
   if ('send_time' in last) {
      result.times.set('send', Number(parseYpDatetime(last.send_time as ProtoDate)));
   }

   return result;
};

export const getProbeFailStatus = (
   status: TStopStatus | TDestroyStatus | TReadinessStatus | TLivenessStatus | null,
): FailStatus => {
   const result: FailStatus = getDefaultFailStatus();
   if (!status) {
      return result;
   }
   if (status.container_status) {
      return getContainerLastFailStatus(status.container_status);
   }

   if ('tcp_check_status' in status) {
      return getTcpHttpUnixLastFailStatus(status.tcp_check_status ?? null);
   }

   if (status.http_get_status) {
      return getTcpHttpUnixLastFailStatus(status.http_get_status);
   }

   if ('unix_signal_status' in status) {
      return getTcpHttpUnixLastFailStatus(status.unix_signal_status ?? null);
   }
   return result;
};
