import { StatusState } from '../../../components/lib';
import { EConditionStatus, TCondition } from '../../../proto-typings';
import { parseYpDatetime, ProtoDate } from '../../../utils';
import {
   EConditionStatus as PodEConditionStatus,
   TCondition as PodTCondition,
} from '../../../proto-typings/yp_proto/yp/client/api/proto/pod_agent';

export enum RuntumeStatusGroup {
   Error = 'error',
   Progress = 'progress',
   Ready = 'ready',
   Other = 'other',
}

const groupOrder: RuntumeStatusGroup[] = [
   RuntumeStatusGroup.Error,
   RuntumeStatusGroup.Progress,
   RuntumeStatusGroup.Other,
   RuntumeStatusGroup.Ready,
];

export const runtimeStatusGroupOrder = new Map(groupOrder.map((group, i) => [group, i]));

export const defaultRuntimeStatusTitle: Record<RuntumeStatusGroup, string> = {
   [RuntumeStatusGroup.Error]: 'failed',
   [RuntumeStatusGroup.Progress]: 'in progress',
   [RuntumeStatusGroup.Ready]: 'ready',
   [RuntumeStatusGroup.Other]: 'unknown',
};

export const runtimeStatusMap: Record<RuntumeStatusGroup, StatusState> = {
   [RuntumeStatusGroup.Ready]: StatusState.Ok,
   [RuntumeStatusGroup.Error]: StatusState.Error,
   [RuntumeStatusGroup.Progress]: StatusState.Progress,
   [RuntumeStatusGroup.Other]: StatusState.Unknown,
};

const groupMap: Partial<Record<StatusState, RuntumeStatusGroup>> = {
   [StatusState.Ok]: RuntumeStatusGroup.Ready,
   [StatusState.Error]: RuntumeStatusGroup.Error,
   [StatusState.Progress]: RuntumeStatusGroup.Progress,
};

export function getRuntimeStatusGroup(state: StatusState): RuntumeStatusGroup {
   return groupMap[state] ?? RuntumeStatusGroup.Other;
}

export interface StatusInfo {
   active: boolean;
   message: string | null;
   reason: string | null;
   rawJSON: string | null;
   lastTransitionTime: number | null;
}

export interface StatusStateInfo {
   ready: StatusInfo;
   inProgress: StatusInfo;
   failed: StatusInfo;
}

export const getDefaultStatusInfo = (): StatusInfo => ({
   active: false,
   message: null,
   reason: null,
   lastTransitionTime: null,
   rawJSON: null,
});

export const getDefaultStatusStateInfo = () =>
   (['ready', 'inProgress', 'failed'] as const).reduce((obj, key) => {
      obj[key] = getDefaultStatusInfo();
      return obj;
   }, {} as StatusStateInfo);

const defaultInfoTitle = {
   ready: 'ready',
   inProgress: 'in progress',
   failed: 'failed',
};

// не во всех местах апи одинаковые enum
type Condition = TCondition | PodTCondition | null;

export interface InfoStatus {
   ready?: Condition;
   failed?: Condition;
   in_progress?: Condition;
}

export function getStatusStateInfo<T extends InfoStatus>(status: T | null): StatusStateInfo {
   const result: StatusStateInfo = getDefaultStatusStateInfo();
   const { ready, in_progress, failed } = status ?? {};
   const statusMap = new Map([
      ['ready', ready],
      ['inProgress', in_progress],
      ['failed', failed],
   ] as const);

   for (const [name, condition] of statusMap) {
      const { message = null, reason, last_transition_time } = condition ?? {};
      const date = last_transition_time ? parseYpDatetime(last_transition_time as ProtoDate) : null;
      result[name] = {
         active:
            condition?.status === PodEConditionStatus.EConditionStatus_TRUE ||
            condition?.status === EConditionStatus.CS_TRUE,
         message,
         lastTransitionTime: date ? Number(date) : null,
         reason: reason?.replaceAll('_', ' ').toLocaleLowerCase() ?? defaultInfoTitle[name],
         rawJSON: condition ? JSON.stringify(condition, null, 2) : null,
      };
   }

   return result;
}

export interface RuntimeState {
   statusGroup: RuntumeStatusGroup;
   message: string;
}

export interface RuntimeStatusData {
   state: RuntimeState | null;
   info: StatusStateInfo | null;
   existMessage: boolean;
   existErrors: boolean;
}

export interface RuntimeStatusResult {
   message: string;
   statusGroup: RuntumeStatusGroup;
   clickable: boolean;
   showWarning: boolean;
}

/**
 * Расчёт приоритетов статусов для отображения, исходя из приоритетов статусов.
 * Контент в модальном окне здесь не определяется, только сам факт наличия.
 */
export function getRuntimeStatus({ state, info, existMessage, existErrors }: RuntimeStatusData): RuntimeStatusResult {
   const { message: stateMessage, statusGroup: stateGroup } = state ?? {};
   const { failed, inProgress, ready } = info ?? {};
   const infoMap = new Map([
      [RuntumeStatusGroup.Error, failed],
      [RuntumeStatusGroup.Progress, inProgress],
      [RuntumeStatusGroup.Ready, ready],
   ]);

   const infoPrimaryGroup = groupOrder.find(group => infoMap.get(group)?.active) ?? null;
   const infoPrimaryData = infoPrimaryGroup ? infoMap.get(infoPrimaryGroup) : null;

   let message: string = existErrors ? 'has errors' : existMessage ? 'has message' : 'unknown';
   let statusGroup: RuntumeStatusGroup = existErrors ? RuntumeStatusGroup.Error : RuntumeStatusGroup.Other;

   const stateOrder = stateGroup ? runtimeStatusGroupOrder.get(stateGroup) ?? +Infinity : +Infinity;
   const infoOrder = infoPrimaryGroup ? runtimeStatusGroupOrder.get(infoPrimaryGroup) ?? +Infinity : +Infinity;
   const isStatePrimary = stateOrder < infoOrder;
   if (stateGroup || infoPrimaryGroup) {
      message = (isStatePrimary ? stateMessage : infoPrimaryData?.reason) ?? 'unknown';
      statusGroup = (isStatePrimary ? stateGroup : infoPrimaryGroup) ?? RuntumeStatusGroup.Other;
   }

   const status: RuntimeStatusResult = {
      message,
      statusGroup,
      clickable: existErrors || existMessage,
      // для ошибки и так понятно, что есть ошибки, для остальных статусов выводим пояснения
      showWarning: existErrors && statusGroup !== RuntumeStatusGroup.Error,
   };

   return status;
}

interface GetUIStatusDataParams<T extends string | number> {
   state: T | null;
   statusMap: Record<T, StatusState>;
   statusStateInfo: StatusStateInfo;
   existMessage?: boolean;
   existErrors?: boolean;
}

export interface GetUIStatusDataResult extends RuntimeStatusResult {
   statusState: StatusState;
}

// данные для отображения статуса непосредственно в компоненте
export function getUIStatusData<T extends string | number>({
   state,
   statusMap,
   statusStateInfo,
   existMessage = false,
   existErrors = false,
}: GetUIStatusDataParams<T>): GetUIStatusDataResult {
   const statusState = state ? statusMap[state] : null;
   const stateGroup = statusState ? getRuntimeStatusGroup(statusState) : null;
   const runtimeStatusData = getRuntimeStatus({
      state: stateGroup ? { statusGroup: stateGroup, message: String(state) } : null,
      info: statusStateInfo,
      existMessage,
      existErrors,
   });
   const { statusGroup } = runtimeStatusData;

   const targetState: StatusState =
      statusGroup === RuntumeStatusGroup.Other ? statusState ?? StatusState.Unknown : runtimeStatusMap[statusGroup];

   const result: GetUIStatusDataResult = {
      ...runtimeStatusData,
      statusState: targetState,
   };

   return result;
}
