import { EMultiSecretFileContentFormat, SecretSelector } from '../../../proto-typings';

/**
 * Ссылка на ключ секрета (отдельную запись в рамках версии секрета)
 */
export interface LinkToSecret {
   /**
    * Ключ секрета (по которому можно вытащить значение)
    *
    * Версия секрета - это слепок пар "ключ-значение". Это поле - ключ.
    *
    * Есть особый вариант с префиксом DUMP_ALL_KEYS: синтаксис "DUMP_ALL_KEYS:<format>", где format - json|java|yaml
    * В этом случае выберутся все ключи в указанном формате (т.н. мультисекрет).
    */
   key: string;

   /**
    * Строка для связки ссылки с хранилищем секретов в DU
    */
   alias: string;
}

/**
 * Преобразует селектор секрета (из спеки) в особый объект-ссылку
 */
export function getLinkToSecret(selector: SecretSelector): LinkToSecret {
   return {
      alias: selector.alias,
      key: selector.id,
   };
}

// region Multisecrets

/**
 * Префикс для ключа в режиме мультисекрета
 */
export const DUMP_ALL_KEYS = 'DUMP_ALL_KEYS';

/**
 * Формат по-умолчанию (если не указан).
 *
 * Значение JSON выяснено эмпирическим путем.
 * Причем, если задать его явно, то оно удалится при сохранении. Придется повторять эту же логику в UI.
 */
export const DEFAULT_MULTISECRET_FORMAT = EMultiSecretFileContentFormat.EMultiSecretFileContentFormat_JSON;

export function isMultisecret(key: string) {
   // TODO временный фикс, есть стейджи с незаданными key, UI падает при попытке редактирования
   return (key ?? '').startsWith(DUMP_ALL_KEYS);
}

export function generateMultisecretKey(format = DEFAULT_MULTISECRET_FORMAT) {
   return `${DUMP_ALL_KEYS}:${format}`;
}

export function extractFormat(key: string): EMultiSecretFileContentFormat | undefined {
   const rawFormat = key.replace(`${DUMP_ALL_KEYS}:`, '') as any;

   return Object.values(EMultiSecretFileContentFormat).includes(rawFormat) && rawFormat !== DEFAULT_MULTISECRET_FORMAT
      ? rawFormat
      : undefined;
}

// endregion
