import { IEnumOption, isEmpty } from '@yandex-infracloud-ui/libs';

import { EnvironmentVariable } from '../../../../modules/environment';
import { EResolvConf } from '../../../../proto-typings';
import { Entity } from '../../../../redux/models';
import { getEmptyWorkload, Workload } from '../Workload';

export const resolvConfOptions: IEnumOption[] = [
   { value: EResolvConf.EResolvConf_DEFAULT, title: EResolvConf.EResolvConf_DEFAULT },
   { value: EResolvConf.EResolvConf_KEEP, title: EResolvConf.EResolvConf_KEEP },
   { value: EResolvConf.EResolvConf_NAT64_LOCAL, title: EResolvConf.EResolvConf_NAT64_LOCAL },
   { value: EResolvConf.EResolvConf_NAT64, title: EResolvConf.EResolvConf_NAT64 },
];

export interface BoxDockerImage {
   enabled: boolean;
   name: string | null;
   tag: string | null;
}

// region Layers

export interface BoxLayer {
   _layerRef: string | null; // ссылка на disk layer
}

// endregion

// region Static resources

export interface BoxStaticResource {
   _order?: number;
   mountPoint: string | null;
   _staticResourceRef: string | null; // ссылка на disk static resource
}

// endregion

// region Dynamic Resources

export enum DynamicResourceNotifyPolicyMode {
   Disabled = 'disabled',
   Exec = 'exec',
   Http = 'http',
}

export interface DynamicResourceNotifyPolicyExecAction {
   commandLine: string | null;
   expectedAnswer: string | null;
}

export interface DynamicResourceNotifyPolicyHttpAction {
   url: string | null;
   expectedAnswer: string | null;
}

export interface DynamicResourceNotifyPolicy {
   mode: DynamicResourceNotifyPolicyMode;
   execAction?: DynamicResourceNotifyPolicyExecAction;
   httpAction?: DynamicResourceNotifyPolicyHttpAction;
}

export interface DynamicResourceVerification {
   checkPeriodMs: number | null;
   checksum: string | null;
}

export interface DynamicResourceAdvancedSettings {
   allowDeduplication: boolean;
   maxDownloadSpeed: number | null;
   verification: DynamicResourceVerification;
}

export interface DynamicResourceCustomSettings {
   deployGroups?: boolean;
   requiredLabels?: boolean;
}

export interface BoxDynamicResource extends Entity {
   initialId?: string;
   urls: (string | null)[];
   destination: string | null;
   storageDir: string | null;
   updateWindow: number | null;
   cachedRevisionsCount: number | null;

   notifyPolicy: DynamicResourceNotifyPolicy;
   advancedSettings: DynamicResourceAdvancedSettings;
   customSettings: DynamicResourceCustomSettings;
}

// endregion

export interface JugglerBundle {
   url: string | null;
   // checksum: string | null;
}

export interface BoxJugglerSettings {
   bundles: JugglerBundle[];
   enabled: boolean;
   port: number | null;
}

export interface LogrotateConfig {
   rawConfig: string | null;
   runPeriodMillisecond: number | null;
}

export enum VolumeMountMode {
   ReadOnly = 'read_only',
   ReadWrite = 'read_write',
}

// @dkochetov: стоит явно семантически различать от режима монтирования
export enum VolumeCreateMode {
   ReadOnly = 'read_only',
   ReadWrite = 'read_write',
}

export const defaultVolumeCreateMode = VolumeCreateMode.ReadWrite;

export enum CgroupFsMountMode {
   None = 'none',
   ReadOnly = 'read_only',
   ReadWrite = 'read_write',
}

export const defaultCgroupFsMountMode = CgroupFsMountMode.ReadOnly;

export interface BoxVolume {
   _order?: number;
   mode: VolumeMountMode;
   mountPoint: string | null;
   _volumeRef: string | null; // ссылка на disk volume
}

export interface BoxRootFsSettings {
   createMode: VolumeCreateMode;
}

export interface Box extends Entity {
   dockerImage: BoxDockerImage;
   cpuPerBox: number | null;
   environment: EnvironmentVariable[];
   juggler: BoxJugglerSettings;
   layers: BoxLayer[];
   ramPerBox: number | null;
   anonymousMemoryLimit: number | null;
   bindSkynet: boolean;
   resolvConf: EResolvConf;
   staticResources: BoxStaticResource[];
   threadLimit: number | null;
   workloads: Workload[];
   dynamicResources: BoxDynamicResource[];

   /**
    * ID данное этому Box при заведении узла в форме.
    * В отличие от id - не редактируется и не меняется, вплоть до сохранения.
    */
   initialId?: string;
   logrotateConfig: LogrotateConfig;

   virtualDiskIdRef: string | null; // ссылка на диск, если дисков несколько
   volumes: BoxVolume[];
   rootFsSettings: BoxRootFsSettings;
   cgroupFsMountMode: CgroupFsMountMode;
}

export const getEmptyBox = (): Box => ({
   cpuPerBox: null,
   dockerImage: {
      enabled: false,
      name: '',
      tag: '',
   },
   environment: [],
   id: 'box',
   juggler: { enabled: false, port: 31579, bundles: [] },
   layers: [], // пустой массив для боксов, не привязанных к emptyDeployUnit с дефолтными слоями
   ramPerBox: null,
   anonymousMemoryLimit: null,
   bindSkynet: false,
   resolvConf: EResolvConf.EResolvConf_DEFAULT,
   staticResources: [],
   dynamicResources: [],
   logrotateConfig: {
      rawConfig: null,
      runPeriodMillisecond: null,
   },
   threadLimit: null,
   workloads: [getEmptyWorkload()],
   virtualDiskIdRef: null,
   volumes: [],
   rootFsSettings: {
      createMode: defaultVolumeCreateMode,
   },
   cgroupFsMountMode: defaultCgroupFsMountMode,
});

export const isBoxAdvancedSettingsEmpty = (advanced: DynamicResourceAdvancedSettings) => {
   const { maxDownloadSpeed, verification } = advanced;

   if (!isEmpty(maxDownloadSpeed)) {
      return false;
   }
   if (!isEmpty(verification)) {
      if (Object.values(verification).some(Boolean)) return false;
   }

   return true;
};
