/* eslint-disable no-underscore-dangle */
import { BYTES } from '@yandex-infracloud-ui/libs';

import {
   EResolvConf,
   EResourceAccessPermissions,
   TBox,
   TDeployUnitSpec,
   TPodAgentSpec,
   TPodResource,
   TPodTemplateSpec,
   TStageSpec,
} from '../../../../proto-typings';
import { hexRegExp, noop } from '../../../../utils';
import {
   DEFAULT_DISK_ID,
   DEFAULT_JUGGLER_PORT,
   DEFAULT_OS_LIST,
   EMPTY_CHECKSUM,
   HDD_BANDWIDTH_LIMIT_FACTOR,
} from '../../../constants';
import { DeepPartial } from '../../../typeHelpers';

import { DeployUnitConverter, DiskType } from '../DeployUnit';

import {
   Box,
   BoxDynamicResource,
   BoxLayer,
   BoxStaticResource,
   BoxVolume,
   DynamicResourceNotifyPolicyMode,
} from './Box';

import { BoxConverter } from './BoxConverter';

const BOX_ID = {
   Default: 'Box-123',
   Docker: 'Box-Docker',
   Juggler: 'Box-Juggler',
};

const DOCKER = {
   name: 'docker',
   tag: '1.2.3',
};

const JUGGLER_CHECKS = ['rbtorrent:juggler1', 'rbtorrent:juggler2', 'rbtorrent:juggler3'];

const LAYER = {
   baseDefault: {
      id: 'base-layer-0',
      url: DEFAULT_OS_LIST[0].value,
      checksum: EMPTY_CHECKSUM,
   },
   baseCustom: {
      id: 'base-layer-1',
      url: 'rbtorrent:custom',
      checksum: EMPTY_CHECKSUM,
   },
   layer: {
      id: 'layer',
      url: 'rbtorrent:layer',
      checksum: EMPTY_CHECKSUM,
   },
   layerWithoutUrl: {
      id: 'layerWithoutUrl',
      // url: undefined,
      checksum: EMPTY_CHECKSUM,
   },
   layerWithoutChecksum: {
      id: 'layerWithoutChecksum',
      url: 'rbtorrent:layerWithoutChecksum',
      // checksum: undefined,
   },
   layerWithCustomChecksum: {
      id: 'layerWithCustomChecksum',
      url: 'rbtorrent:layerWithCustomChecksum',
      checksum: 'CUSTOM_CHECKSUM',
   },
};

const STATIC_RESOURCES = ({
   resourceUrl: {
      id: 'resourceUrl',
      url: 'rbtorrent:resource1',
      verification: {
         checksum: EMPTY_CHECKSUM,
      },
      access_permissions: EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED,
   },
   resourceUrlWithoutChecksum: {
      id: 'resourceWithoutChecksum',
      url: 'rbtorrent:resourceWithoutChecksum',
      access_permissions: EResourceAccessPermissions.EResourceAccessPermissions_600,
   },
   resourceUrlWithCustomChecksum: {
      id: 'resource-1',
      url: 'rbtorrent:resource1',
      verification: {
         checksum: 'CUSTOM_CHECKSUM',
      },
      access_permissions: EResourceAccessPermissions.EResourceAccessPermissions_660,
   },
   resourceFiles: {
      id: 'resourceFiles',
      files: {
         files: [
            {
               file_name: 'raw',
               raw_data: 'raw data',
            },
            {
               file_name: 'secret',
               secret_data: {
                  alias: 'sec-123:ver-123',
                  id: 'key',
               },
            },
            {
               file_name: 'multi_secret',
               multi_secret_data: {
                  secret_alias: 'sec-123:ver-123',
               },
            },
         ],
      },
      verification: {
         checksum: EMPTY_CHECKSUM,
      },
      // access_permissions: undefined
   },
   unknownResource: {
      id: 'unknownResource',
      verification: {
         checksum: EMPTY_CHECKSUM,
      },
      // access_permissions: undefined
   },
} as unknown) as Record<string, TPodResource>;

const getInitialBoxSpec = (): DeepPartial<TBox> => ({
   id: BOX_ID.Default,
});

const getInitialPodAgentSpec = (): DeepPartial<TPodAgentSpec> => ({
   // ...
   boxes: [
      getInitialBoxSpec(),
      {
         ...getInitialBoxSpec(),
         id: BOX_ID.Docker,
      },
      {
         ...getInitialBoxSpec(),
         id: BOX_ID.Juggler,
      },
   ],
   resources: {
      layers: [
         LAYER.baseDefault,
         LAYER.baseCustom,
         LAYER.layer,
         LAYER.layerWithoutUrl,
         LAYER.layerWithoutChecksum,
         LAYER.layerWithCustomChecksum,
      ],
      static_resources: [
         STATIC_RESOURCES.resourceUrl,
         STATIC_RESOURCES.resourceUrlWithoutChecksum,
         STATIC_RESOURCES.resourceUrlWithCustomChecksum,
         STATIC_RESOURCES.resourceFiles,
         STATIC_RESOURCES.unknownResource,
      ],
   },
});

const getInitialPodTemplateSpec = (): DeepPartial<TPodTemplateSpec> => ({
   spec: {
      host_infra: {
         monitoring: {},
      },
      pod_agent_payload: {
         spec: getInitialPodAgentSpec(),
      },
      disk_volume_requests: [
         {
            id: DEFAULT_DISK_ID,
            storage_class: DiskType.HDD,
            quota_policy: {
               capacity: 15 * BYTES.GB,
               bandwidth_guarantee: 25 * BYTES.MB,
               bandwidth_limit: 100 * BYTES.MB * HDD_BANDWIDTH_LIMIT_FACTOR,
            },
            labels: {
               x: 3,
               used_by_infra: true,
            },
         },
      ],
   },
});

const getInitialDeployUnitSpec = (): DeepPartial<TDeployUnitSpec> => ({
   images_for_boxes: {
      [BOX_ID.Docker]: {
         registry_host: 'registry.yandex.net',
         name: DOCKER.name,
         tag: DOCKER.tag,
      },
   },
   logrotate_configs: {
      [BOX_ID.Default]: {
         raw_config: 'raw_config',
         run_period_millisecond: 3600000,
      },
   },
   box_juggler_configs: {
      [BOX_ID.Juggler]: {
         port: DEFAULT_JUGGLER_PORT,
         archived_checks: [
            {
               url: JUGGLER_CHECKS[0],
            },
            {
               url: JUGGLER_CHECKS[1],
            },
            {
               url: JUGGLER_CHECKS[2],
            },
         ],
      },
   },
   replica_set: {
      // ...
      replica_set_template: {
         pod_template_spec: getInitialPodTemplateSpec(),
      },
   },
});

const getDynamicResourcesSpec = (): DeepPartial<TStageSpec['dynamic_resources']> => ({
   customDeployGroups: {
      deploy_unit_ref: 'any',
      dynamic_resource: {
         deploy_groups: [
            {
               mark: 'all',
               storage_options: {
                  box_ref: 'any',
                  destination: '/destination/any',
                  storage_dir: '/storage/any',
                  cached_revisions_count: 1,
               },
               urls: ['sbr:1234567890'],
            },
            {
               mark: 'all',
               storage_options: {
                  box_ref: BOX_ID.Default,
                  destination: '/destination/customDeployGroups',
                  storage_dir: '/storage/customDeployGroups',
                  cached_revisions_count: 5,
               },
               urls: ['sbr:1234567890'],
            },
         ],
         // revision: 99,
         update_window: 1,
      },
   },
   customRequiredLabels: {
      deploy_unit_ref: 'any',
      dynamic_resource: {
         deploy_groups: [
            {
               mark: 'all',
               storage_options: {
                  box_ref: BOX_ID.Default,
                  destination: '/destination/customRequiredLabels',
                  storage_dir: '/storage/customRequiredLabels',
                  cached_revisions_count: 10,
               },
               urls: ['sbr:1234567890'],
               required_labels: {
                  label1: 'label1',
                  label2: 'label2',
               },
            },
         ],
         // revision: 99,
         update_window: 2,
      },
   },
   httpAction: {
      deploy_unit_ref: 'any',
      dynamic_resource: {
         deploy_groups: [
            {
               mark: 'all',
               storage_options: {
                  box_ref: BOX_ID.Default,
                  destination: '/destination/http',
                  storage_dir: '/storage/http',
                  cached_revisions_count: 3,
                  http_action: {
                     url: 'http://url...',
                     expected_answer: 'http answer',
                  },
               },
               urls: ['sbr:1234567890'],
            },
         ],
         // revision: 99,
         update_window: 2,
      },
   },
   execAction: {
      deploy_unit_ref: 'any',
      dynamic_resource: {
         deploy_groups: [
            {
               mark: 'all',
               storage_options: {
                  box_ref: BOX_ID.Default,
                  destination: '/destination/exec',
                  storage_dir: '/storage/exec',
                  cached_revisions_count: 3,
                  exec_action: {
                     command_line: 'exec command',
                     expected_answer: 'exec answer',
                  },
               },
               urls: ['sbr:1234567890'],
            },
         ],
         // revision: 99,
         update_window: 2,
      },
   },
   disabledAction: {
      deploy_unit_ref: 'any',
      dynamic_resource: {
         deploy_groups: [
            {
               mark: 'all',
               storage_options: {
                  box_ref: BOX_ID.Default,
                  destination: '/destination/disabled',
                  storage_dir: '/storage/disabled',
                  allow_deduplication: true,
                  cached_revisions_count: 15,
                  max_download_speed: 50,
                  verification: {
                     check_period_ms: 1000,
                     checksum: 'checksum',
                  },
               },
               urls: ['sbr:1234567890', 'rbtorrent:abc1234567890...'],
            },
         ],
         // revision: 99,
         update_window: 2,
      },
   },
   empty: {
      deploy_unit_ref: 'any',
      dynamic_resource: {
         deploy_groups: [
            {
               storage_options: {
                  box_ref: BOX_ID.Default,
                  // ...
               },
               // ...
            },
         ],
         // revision: 99,
         // ...
      },
   },
   unused: {
      deploy_unit_ref: 'any',
      dynamic_resource: {
         deploy_groups: [
            {
               mark: 'all',
               storage_options: {
                  box_ref: 'unused',
                  // ...
               },
               // ...
            },
         ],
         // revision: 99,
         // ...
      },
   },
});

const podTemplateSpec = getInitialPodTemplateSpec() as TPodTemplateSpec;
const deployUnitSpec = getInitialDeployUnitSpec() as TDeployUnitSpec;
const dynamicResources = getDynamicResourcesSpec() as TStageSpec['dynamic_resources'];
const disks = DeployUnitConverter.getDisks(podTemplateSpec);

/**
 * @example
 * it('should xxx', check(
 *    spec => {
 *
 *    },
 *    expected => {
 *
 *    },
 *    model => {
 *
 *    }
 *  ))
 */
const check = (
   patchSpecBefore: (s: DeepPartial<TBox>) => void,
   patchExpectedModel: (s: DeepPartial<Box>) => void,
   checkOnlyBranch?: (s: Box) => any,
) => () => {
   const spec = getInitialBoxSpec() as TBox;
   patchSpecBefore(spec);

   const model = BoxConverter.fromApi(spec, podTemplateSpec, deployUnitSpec, dynamicResources, disks);

   const expectedConvertedModel = getInitialBoxSpec();
   patchExpectedModel(expectedConvertedModel as any); // TODO typing

   if (checkOnlyBranch) {
      expect(checkOnlyBranch(model)).toEqual(checkOnlyBranch(expectedConvertedModel as any)); // TODO typing
   } else {
      expect(model).toEqual(expectedConvertedModel);
   }
};

describe('models/ui|BoxConverter', () => {
   it(
      'should extract id',
      check(
         noop,
         expected => {
            expected.id = BOX_ID.Default;
         },
         model => model.id,
      ),
   );

   it(
      'should extract id',
      check(
         spec => {
            spec.id = 'CustomBoxId';
         },
         expected => {
            expected.id = 'CustomBoxId';
         },
         model => model.id,
      ),
   );

   it(
      'should extract cpuPerBox: empty',
      check(
         noop,
         expected => {
            expected.cpuPerBox = null;
         },
         model => model.cpuPerBox,
      ),
   );

   it(
      'should extract cpuPerBox',
      check(
         spec => {
            spec.compute_resources = {
               vcpu_limit: 100,
            };
         },
         expected => {
            expected.cpuPerBox = 100;
         },
         model => model.cpuPerBox,
      ),
   );

   it(
      'should extract ramPerBox: empty',
      check(
         noop,
         expected => {
            expected.ramPerBox = null;
         },
         model => model.ramPerBox,
      ),
   );

   it(
      'should extract ramPerBox',
      check(
         spec => {
            spec.compute_resources = {
               memory_limit: BYTES.GB,
            };
         },
         expected => {
            expected.ramPerBox = BYTES.GB;
         },
         model => model.ramPerBox,
      ),
   );

   it(
      'should extract anonymousMemoryLimit: empty',
      check(
         noop,
         expected => {
            expected.anonymousMemoryLimit = null;
         },
         model => model.anonymousMemoryLimit,
      ),
   );

   it(
      'should extract anonymousMemoryLimit',
      check(
         spec => {
            spec.compute_resources = {
               anonymous_memory_limit: BYTES.GB,
            };
         },
         expected => {
            expected.anonymousMemoryLimit = BYTES.GB;
         },
         model => model.anonymousMemoryLimit,
      ),
   );

   it(
      'should extract threadLimit: empty',
      check(
         noop,
         expected => {
            expected.threadLimit = null;
         },
         model => model.threadLimit,
      ),
   );

   it(
      'should extract threadLimit',
      check(
         spec => {
            spec.compute_resources = {
               thread_limit: 200,
            };
         },
         expected => {
            expected.threadLimit = 200;
         },
         model => model.threadLimit,
      ),
   );

   it(
      'should extract resolvConf: default',
      check(
         noop,
         expected => {
            expected.resolvConf = EResolvConf.EResolvConf_DEFAULT;
         },
         model => model.resolvConf,
      ),
   );

   it(
      'should extract resolvConf: keep',
      check(
         spec => {
            spec.resolv_conf = EResolvConf.EResolvConf_KEEP;
         },
         expected => {
            expected.resolvConf = EResolvConf.EResolvConf_KEEP;
         },
         model => model.resolvConf,
      ),
   );

   it(
      'should extract resolvConf: nat64',
      check(
         spec => {
            spec.resolv_conf = EResolvConf.EResolvConf_NAT64;
         },
         expected => {
            expected.resolvConf = EResolvConf.EResolvConf_NAT64;
         },
         model => model.resolvConf,
      ),
   );

   it(
      'should extract bind_skynet: empty',
      check(
         noop,
         expected => {
            expected.bindSkynet = false;
         },
         model => model.bindSkynet,
      ),
   );

   it(
      'should extract bind_skynet: false',
      check(
         spec => {
            spec.bind_skynet = false;
         },
         expected => {
            expected.bindSkynet = false;
         },
         model => model.bindSkynet,
      ),
   );

   it(
      'should extract bind_skynet: true',
      check(
         spec => {
            spec.bind_skynet = true;
         },
         expected => {
            expected.bindSkynet = true;
         },
         model => model.bindSkynet,
      ),
   );

   it(
      'should extract docker',
      check(
         spec => {
            spec.id = BOX_ID.Docker;
         },
         expected => {
            expected.dockerImage = {
               enabled: true,
               name: DOCKER.name,
               tag: DOCKER.tag,
            };
         },
         model => model.dockerImage,
      ),
   );

   it(
      'should extract dynamic resources: empty',
      check(
         spec => {
            spec.id = BOX_ID.Juggler;
         },
         expected => {
            expected.dynamicResources = [] as BoxDynamicResource[];
         },
         model => model.dynamicResources,
      ),
   );

   it(
      'should extract dynamic resources',
      check(
         noop,
         expected => {
            expected.dynamicResources = [
               {
                  id: 'customDeployGroups',
                  initialId: 'customDeployGroups',
                  urls: ['sbr:1234567890'],
                  destination: '/destination/customDeployGroups',
                  storageDir: '/storage/customDeployGroups',
                  updateWindow: 1,
                  cachedRevisionsCount: 5,

                  notifyPolicy: {
                     mode: DynamicResourceNotifyPolicyMode.Disabled,
                  },

                  advancedSettings: {
                     allowDeduplication: false,
                     maxDownloadSpeed: null,
                     verification: {
                        checkPeriodMs: null,
                        checksum: null,
                     },
                  },

                  customSettings: {
                     deployGroups: true,
                     requiredLabels: false,
                  },
               },
               {
                  id: 'customRequiredLabels',
                  initialId: 'customRequiredLabels',
                  urls: ['sbr:1234567890'],
                  destination: '/destination/customRequiredLabels',
                  storageDir: '/storage/customRequiredLabels',
                  updateWindow: 2,
                  cachedRevisionsCount: 10,

                  notifyPolicy: {
                     mode: DynamicResourceNotifyPolicyMode.Disabled,
                  },

                  advancedSettings: {
                     allowDeduplication: false,
                     maxDownloadSpeed: null,
                     verification: {
                        checkPeriodMs: null,
                        checksum: null,
                     },
                  },

                  customSettings: {
                     deployGroups: false,
                     requiredLabels: true,
                  },
               },
               {
                  id: 'httpAction',
                  initialId: 'httpAction',
                  urls: ['sbr:1234567890'],
                  destination: '/destination/http',
                  storageDir: '/storage/http',
                  updateWindow: 2,
                  cachedRevisionsCount: 3,

                  notifyPolicy: {
                     mode: DynamicResourceNotifyPolicyMode.Http,
                     httpAction: {
                        url: 'http://url...',
                        expectedAnswer: 'http answer',
                     },
                  },

                  advancedSettings: {
                     allowDeduplication: false,
                     maxDownloadSpeed: null,
                     verification: {
                        checkPeriodMs: null,
                        checksum: null,
                     },
                  },

                  customSettings: {
                     deployGroups: false,
                     requiredLabels: false,
                  },
               },
               {
                  id: 'execAction',
                  initialId: 'execAction',
                  urls: ['sbr:1234567890'],
                  destination: '/destination/exec',
                  storageDir: '/storage/exec',
                  updateWindow: 2,
                  cachedRevisionsCount: 3,

                  notifyPolicy: {
                     mode: DynamicResourceNotifyPolicyMode.Exec,
                     execAction: {
                        commandLine: 'exec command',
                        expectedAnswer: 'exec answer',
                     },
                  },

                  advancedSettings: {
                     allowDeduplication: false,
                     maxDownloadSpeed: null,
                     verification: {
                        checkPeriodMs: null,
                        checksum: null,
                     },
                  },

                  customSettings: {
                     deployGroups: false,
                     requiredLabels: false,
                  },
               },
               {
                  id: 'disabledAction',
                  initialId: 'disabledAction',
                  urls: ['sbr:1234567890', 'rbtorrent:abc1234567890...'],
                  destination: '/destination/disabled',
                  storageDir: '/storage/disabled',
                  updateWindow: 2,
                  cachedRevisionsCount: 15,

                  notifyPolicy: {
                     mode: DynamicResourceNotifyPolicyMode.Disabled,
                  },

                  advancedSettings: {
                     allowDeduplication: true,
                     maxDownloadSpeed: 50,
                     verification: {
                        checkPeriodMs: 1000,
                        checksum: 'checksum',
                     },
                  },

                  customSettings: {
                     deployGroups: false,
                     requiredLabels: false,
                  },
               },
               {
                  id: 'empty',
                  initialId: 'empty',
                  urls: [null],
                  destination: null,
                  storageDir: null,
                  updateWindow: null,
                  cachedRevisionsCount: null,

                  notifyPolicy: {
                     mode: DynamicResourceNotifyPolicyMode.Disabled,
                  },

                  advancedSettings: {
                     allowDeduplication: false,
                     maxDownloadSpeed: null,
                     verification: {
                        checkPeriodMs: null,
                        checksum: null,
                     },
                  },

                  customSettings: {
                     deployGroups: false,
                     requiredLabels: false,
                  },
               },
            ] as BoxDynamicResource[];
         },
         model => model.dynamicResources,
      ),
   );

   it(
      'should extract juggler: empty',
      check(
         noop,
         expected => {
            expected.juggler = {
               enabled: false,
               port: DEFAULT_JUGGLER_PORT,
               bundles: [],
            };
         },
         model => model.juggler,
      ),
   );

   it(
      'should extract juggler',
      check(
         spec => {
            spec.id = BOX_ID.Juggler;
         },
         expected => {
            expected.juggler = {
               enabled: true,
               port: DEFAULT_JUGGLER_PORT,
               bundles: [{ url: JUGGLER_CHECKS[0] }, { url: JUGGLER_CHECKS[1] }, { url: JUGGLER_CHECKS[2] }],
            };
         },
         model => model.juggler,
      ),
   );

   it(
      'should extract logrotate_configs: empty',
      check(
         spec => {
            spec.id = BOX_ID.Juggler;
         },
         expected => {
            expected.logrotateConfig = {
               rawConfig: null,
               runPeriodMillisecond: null,
            };
         },
         model => model.logrotateConfig,
      ),
   );

   it(
      'should extract logrotate_configs',
      check(
         noop,
         expected => {
            expected.logrotateConfig = {
               rawConfig: 'raw_config',
               runPeriodMillisecond: 3600000,
            };
         },
         model => model.logrotateConfig,
      ),
   );

   it(
      'should extract workloads: empty',
      check(
         noop,
         expected => {
            expected.workloads = [];
         },
         model => model.workloads,
      ),
   );
});

describe('Box resources', () => {
   it(
      'should extract layers: empty',
      check(
         spec => {
            spec.id = BOX_ID.Docker;
         },
         expected => {
            expected.layers = [];
         },
         model => model.layers,
      ),
   );

   it('should extract layers', () => {
      const spec = getInitialBoxSpec() as TBox;

      spec.rootfs = {
         layer_refs: [
            LAYER.baseCustom.id,
            LAYER.layer.id,
            LAYER.layerWithoutUrl.id,
            LAYER.layerWithoutChecksum.id,
            LAYER.layerWithCustomChecksum.id,
         ],
      } as TBox['rootfs'];

      const model = BoxConverter.fromApi(spec, podTemplateSpec, deployUnitSpec, dynamicResources, disks);

      const layerRef0 = model.layers[0]._layerRef;
      const layerRef1 = model.layers[1]._layerRef;
      const layerRef2 = model.layers[2]._layerRef;
      const layerRef3 = model.layers[3]._layerRef;
      const layerRef4 = model.layers[4]._layerRef;

      expect(layerRef0).toMatch(hexRegExp);
      expect(layerRef1).toMatch(hexRegExp);
      expect(layerRef2).toMatch(hexRegExp);
      expect(layerRef3).toMatch(hexRegExp);
      expect(layerRef4).toMatch(hexRegExp);

      const expected: BoxLayer[] = [
         {
            _layerRef: layerRef0,
         },
         {
            _layerRef: layerRef1,
         },
         {
            _layerRef: layerRef2,
         },
         {
            _layerRef: layerRef3,
         },
         {
            _layerRef: layerRef4,
         },
      ];

      expect(model.layers).toEqual(expected);
   });

   it(
      'should extract static resources: empty',
      check(
         noop,
         expected => {
            expected.staticResources = [];
         },
         model => model.staticResources,
      ),
   );

   it('should extract static resources', () => {
      const spec = getInitialBoxSpec() as TBox;

      spec.static_resources = [
         {
            resource_ref: STATIC_RESOURCES.resourceUrl.id,
            mount_point: 'mount1',
         },
         {
            resource_ref: STATIC_RESOURCES.resourceUrlWithoutChecksum.id,
            mount_point: 'mount2',
         },
         {
            resource_ref: STATIC_RESOURCES.resourceUrlWithCustomChecksum.id,
            mount_point: 'mount3',
         },
         {
            resource_ref: STATIC_RESOURCES.resourceFiles.id,
            mount_point: 'mount4',
         },
         {
            resource_ref: STATIC_RESOURCES.unknownResource.id,
            mount_point: 'mount5',
         },
      ] as TBox['static_resources'];

      const model = BoxConverter.fromApi(spec, podTemplateSpec, deployUnitSpec, dynamicResources, disks);

      const staticResourceRef0 = model.staticResources[0]._staticResourceRef;
      const staticResourceRef1 = model.staticResources[1]._staticResourceRef;
      const staticResourceRef2 = model.staticResources[2]._staticResourceRef;
      const staticResourceRef3 = model.staticResources[3]._staticResourceRef;
      const staticResourceRef4 = model.staticResources[4]._staticResourceRef;

      expect(staticResourceRef0).toMatch(hexRegExp);
      expect(staticResourceRef1).toMatch(hexRegExp);
      expect(staticResourceRef2).toMatch(hexRegExp);
      expect(staticResourceRef3).toMatch(hexRegExp);
      expect(staticResourceRef4).toMatch(hexRegExp);

      const expected: BoxStaticResource[] = [
         {
            _order: 0,
            mountPoint: 'mount1',
            _staticResourceRef: staticResourceRef0,
         },
         {
            _order: 1,
            mountPoint: 'mount2',
            _staticResourceRef: staticResourceRef1,
         },
         {
            _order: 2,
            mountPoint: 'mount3',
            _staticResourceRef: staticResourceRef2,
         },
         {
            _order: 3,
            mountPoint: 'mount4',
            _staticResourceRef: staticResourceRef3,
         },
         {
            _order: 4,
            mountPoint: 'mount5',
            _staticResourceRef: staticResourceRef4,
         },
      ];

      expect(model.staticResources).toEqual(expected);
   });

   it(
      'should extract volumes: empty',
      check(
         noop,
         expected => {
            expected.volumes = [] as BoxVolume[];
         },
         model => model.volumes,
      ),
   );

   // TODO
   // it(
   //    'should extract volumes',
   //    check(
   //       spec => {
   //          // ...
   //       },
   //       expected => {
   //          expected.volumes = [] as BoxVolume[];
   //       },
   //       model => model.volumes,
   //    ),
   // );
});
