import { IJsonable, isEmpty } from '@yandex-infracloud-ui/libs';

import {
   convertEnvironment,
   EnvironmentVariable,
   getSecretUsagesFromEnvironment,
} from '../../../../modules/environment';
import {
   ECgroupFsMountMode,
   EResolvConf,
   EVolumeCreateMode,
   EVolumeMountMode,
   TBox,
   TDeployUnitSpec,
   TPodAgentSpec,
   TPodTemplateSpec,
   TStageSpec,
} from '../../../../proto-typings';
import { DEFAULT_JUGGLER_PORT } from '../../../constants';
import { getSecretUsagePath, SecretUsage } from '../../secrets';

import { DeployUnitDisk } from '../DeployUnit';
import { Workload, WorkloadConverter } from '../Workload';
import {
   Box,
   BoxDockerImage,
   BoxDynamicResource,
   BoxJugglerSettings,
   BoxLayer,
   BoxRootFsSettings,
   BoxStaticResource,
   BoxVolume,
   CgroupFsMountMode,
   defaultVolumeCreateMode,
   DynamicResourceNotifyPolicyMode,
   LogrotateConfig,
   VolumeCreateMode,
   VolumeMountMode,
} from './Box';

const cgroupfsModes: Record<ECgroupFsMountMode, CgroupFsMountMode> = {
   [ECgroupFsMountMode.ECgroupFsMountMode_NONE]: CgroupFsMountMode.None,
   [ECgroupFsMountMode.ECgroupFsMountMode_RO]: CgroupFsMountMode.ReadOnly,
   [ECgroupFsMountMode.ECgroupFsMountMode_RW]: CgroupFsMountMode.ReadWrite,
};

const rootfsCreateModes: Record<EVolumeCreateMode, VolumeCreateMode> = {
   [EVolumeCreateMode.EVolumeCreateMode_READ_ONLY]: VolumeCreateMode.ReadOnly,
   [EVolumeCreateMode.EVolumeCreateMode_READ_WRITE]: VolumeCreateMode.ReadWrite,
};

export class BoxConverter implements Box, IJsonable {
   public static fromApi(
      raw: TBox,
      podTemplateSpec: TPodTemplateSpec,
      deployUnitSpec: TDeployUnitSpec,
      deployUnitDynamicResources: TStageSpec['dynamic_resources'],
      disks: DeployUnitDisk[],
   ): Box {
      return new BoxConverter(raw, podTemplateSpec, deployUnitSpec, deployUnitDynamicResources, disks).toJSON();
   }

   public static getSecretUsages(box: Box, alias: string | null): SecretUsage[] {
      const result: SecretUsage[] = [];

      // Environment variables
      const environmentPathPrefix = getSecretUsagePath({
         field: 'environment',
         boxId: box.id,
      });

      result.push(...getSecretUsagesFromEnvironment(environmentPathPrefix, box.environment, alias));

      // From workloads
      for (const workload of box.workloads) {
         result.push(...WorkloadConverter.getSecretUsages(box.id, workload, alias));
      }

      return result;
   }

   public cpuPerBox: number | null;

   public dockerImage: BoxDockerImage;

   public id: string;

   public juggler: BoxJugglerSettings;

   public layers: BoxLayer[];

   public logrotateConfig: LogrotateConfig;

   public ramPerBox: number | null;

   public anonymousMemoryLimit: number | null;

   public bindSkynet: boolean;

   public resolvConf: EResolvConf;

   public staticResources: BoxStaticResource[];

   public dynamicResources: BoxDynamicResource[];

   public threadLimit: number | null;

   public workloads: Workload[];

   public environment: EnvironmentVariable[];

   public virtualDiskIdRef: string | null;

   public volumes: BoxVolume[];

   public rootFsSettings: BoxRootFsSettings;

   public cgroupFsMountMode: CgroupFsMountMode;

   constructor(
      private raw: TBox,
      // TODO: это место выглядит неконсистентно
      // мы то this используем, то прокидываем параметры
      // может стоит причесать все конструкторы?
      podTemplateSpec: TPodTemplateSpec,
      deployUnitSpec: TDeployUnitSpec,
      deployUnitDynamicResources: TStageSpec['dynamic_resources'],
      disks: DeployUnitDisk[],
   ) {
      this.id = this.raw.id;
      this.bindSkynet = this.raw.bind_skynet ?? false;
      this.resolvConf = this.raw.resolv_conf ?? EResolvConf.EResolvConf_DEFAULT;
      this.juggler = this.getJuggler(deployUnitSpec);
      this.environment = convertEnvironment(this.raw.env);

      this.logrotateConfig = this.getLogrotateConfig(deployUnitSpec);

      // compute_resources
      this.cpuPerBox = this.raw.compute_resources?.vcpu_limit ?? null;
      this.ramPerBox = this.raw.compute_resources?.memory_limit ?? null;
      this.anonymousMemoryLimit = this.raw.compute_resources?.anonymous_memory_limit ?? null;
      this.threadLimit = this.raw.compute_resources?.thread_limit ?? null;

      const agentSpec = podTemplateSpec.spec?.pod_agent_payload?.spec;

      // eslint-disable-next-line no-underscore-dangle
      this.virtualDiskIdRef =
         this.raw.virtual_disk_id_ref ?? podTemplateSpec.spec?.disk_volume_requests?.[0]?.id ?? null;

      // eslint-disable-next-line no-underscore-dangle
      const disk = disks.find(v => v.id === this.virtualDiskIdRef);
      // region Resources
      this.dockerImage = this.getDocker(deployUnitSpec);
      this.layers = this.getLayers(disk);
      this.staticResources = this.getStaticResources(disk);
      this.dynamicResources = this.getDynamicResources(deployUnitDynamicResources);
      // endregion

      // children
      this.workloads = this.getWorkloads(agentSpec, podTemplateSpec, deployUnitSpec);

      this.volumes = this.getVolumes(disks);

      this.rootFsSettings = this.getRootFsSettings();

      this.cgroupFsMountMode = this.getCgroupFsMountMode();
   }

   public toJSON(): Box {
      return {
         cpuPerBox: this.cpuPerBox,
         dockerImage: this.dockerImage,
         environment: this.environment,
         id: this.id,
         juggler: this.juggler,
         layers: this.layers,
         logrotateConfig: this.logrotateConfig,
         ramPerBox: this.ramPerBox,
         anonymousMemoryLimit: this.anonymousMemoryLimit,
         bindSkynet: this.bindSkynet,
         resolvConf: this.resolvConf,
         staticResources: this.staticResources,
         dynamicResources: this.dynamicResources,
         threadLimit: this.threadLimit,
         workloads: this.workloads,
         // eslint-disable-next-line no-underscore-dangle
         virtualDiskIdRef: this.virtualDiskIdRef,
         volumes: this.volumes,
         rootFsSettings: this.rootFsSettings,
         cgroupFsMountMode: this.cgroupFsMountMode,
      };
   }

   private getCgroupFsMountMode(): CgroupFsMountMode {
      const rawMode = this.raw.cgroup_fs_mount_mode;

      if (rawMode && cgroupfsModes[rawMode]) {
         return cgroupfsModes[rawMode];
      }

      return cgroupfsModes.none; // None, если в спеке пусто
   }

   private getDynamicResources(deployUnitDynamicResources: TStageSpec['dynamic_resources']) {
      const boxDynamicResources: BoxDynamicResource[] = [];

      if (deployUnitDynamicResources) {
         for (const key in deployUnitDynamicResources) {
            if (deployUnitDynamicResources.hasOwnProperty(key)) {
               const resource = deployUnitDynamicResources[key].dynamic_resource;
               const deployGroups = resource?.deploy_groups ?? [];

               if (!isEmpty(deployGroups)) {
                  const boxDeployGroup = deployGroups.find(v => v.storage_options?.box_ref === this.id);

                  if (boxDeployGroup) {
                     const customSettings = {
                        deployGroups: deployGroups.length > 1,
                        requiredLabels: !isEmpty(boxDeployGroup.required_labels ?? {}) ?? false,
                     };

                     const storageOptions = boxDeployGroup.storage_options;

                     const execAction = storageOptions?.exec_action;
                     const httpAction = storageOptions?.http_action;

                     const boxResource = {
                        id: key,
                        initialId: key,

                        // custom settings:
                        // не даем редактировать, если:
                        // deploy_groups.length > 1
                        // required_labels.length > 0
                        customSettings,

                        urls: boxDeployGroup.urls ?? [null],
                        destination: storageOptions?.destination ?? null,
                        storageDir: storageOptions?.storage_dir ?? null,
                        updateWindow: resource!.update_window ?? null,
                        cachedRevisionsCount: storageOptions?.cached_revisions_count ?? null,

                        notifyPolicy: {
                           mode: execAction
                              ? DynamicResourceNotifyPolicyMode.Exec
                              : httpAction
                              ? DynamicResourceNotifyPolicyMode.Http
                              : DynamicResourceNotifyPolicyMode.Disabled,

                           execAction: execAction && {
                              commandLine: execAction.command_line ?? null,
                              expectedAnswer: execAction.expected_answer ?? null,
                           },

                           httpAction: httpAction && {
                              url: httpAction.url ?? null,
                              expectedAnswer: httpAction.expected_answer ?? null,
                           },
                        },

                        advancedSettings: {
                           allowDeduplication: storageOptions?.allow_deduplication ?? false,
                           maxDownloadSpeed: storageOptions?.max_download_speed ?? null,

                           verification: {
                              checksum: storageOptions?.verification?.checksum ?? null,
                              checkPeriodMs: storageOptions?.verification?.check_period_ms ?? null,
                           },
                        },
                     };

                     boxDynamicResources.push(boxResource);
                  }
               }
            }
         }
      }
      return boxDynamicResources;
   }

   private getJuggler(deployUnitSpec: TDeployUnitSpec): BoxJugglerSettings {
      const jugglerConfig = deployUnitSpec.box_juggler_configs?.[this.id];
      if (!jugglerConfig) {
         return {
            enabled: false,
            port: DEFAULT_JUGGLER_PORT,
            bundles: [],
         };
      }

      return {
         enabled: true,
         port: jugglerConfig.port ?? null,
         bundles:
            jugglerConfig.archived_checks?.map(v => ({
               url: v.url,
            })) ?? [],
      };
   }

   private getVolumes(disks: DeployUnitDisk[]): BoxVolume[] {
      const disksVolumes = disks.flatMap(v => v.volumes);

      if (isEmpty(disksVolumes)) {
         return [];
      }

      const boxVolumes: BoxVolume[] = [];

      this.raw.volumes?.forEach((volume, i) => {
         const diskVolume = disksVolumes.find(v => v.id === volume.volume_ref);

         if (diskVolume) {
            boxVolumes.push({
               _order: i,
               mode:
                  volume.mode === EVolumeMountMode.EVolumeMountMode_READ_WRITE
                     ? VolumeMountMode.ReadWrite
                     : VolumeMountMode.ReadOnly,
               mountPoint: volume.mount_point ?? null,
               // eslint-disable-next-line no-underscore-dangle
               _volumeRef: diskVolume._ref,
            });
         }
      });

      return boxVolumes;
   }

   private getDocker(deployUnitSpec: TDeployUnitSpec): BoxDockerImage {
      const dockerForBox = deployUnitSpec.images_for_boxes?.[this.id];

      return {
         enabled: dockerForBox !== undefined && !isEmpty(dockerForBox),
         name: dockerForBox?.name ?? null,
         tag: dockerForBox?.tag ?? null,
      };
   }

   private getLayers(disk: DeployUnitDisk | undefined): BoxLayer[] {
      const boxLayers: BoxLayer[] = [];

      if (disk) {
         (this.raw.rootfs?.layer_refs ?? []).forEach(id => {
            const boxLayer = disk.layers.find(v => v.id === id);

            if (boxLayer) {
               boxLayers.push({
                  // eslint-disable-next-line no-underscore-dangle
                  _layerRef: boxLayer?._ref,
               });
            }
         });
      }

      return boxLayers;
   }

   private getLogrotateConfig(deployUnitSpec: TDeployUnitSpec): LogrotateConfig {
      const logrotateConfig = deployUnitSpec.logrotate_configs?.[this.id] ?? {};

      return {
         rawConfig: logrotateConfig.raw_config ?? null,
         runPeriodMillisecond: logrotateConfig.run_period_millisecond ?? null,
      };
   }

   private getRootFsSettings(): BoxRootFsSettings {
      const rawRootFs = this.raw.rootfs;
      const { create_mode } = rawRootFs ?? {};
      let createMode: VolumeCreateMode = defaultVolumeCreateMode;
      if (create_mode) {
         createMode = rootfsCreateModes[create_mode] ?? defaultVolumeCreateMode;
      }
      const settings: BoxRootFsSettings = {
         createMode,
      };

      return settings;
   }

   private getStaticResources(disk: DeployUnitDisk | undefined): BoxStaticResource[] {
      const boxStaticResources: BoxStaticResource[] = [];

      if (disk) {
         (this.raw.static_resources ?? []).forEach((resource, i) => {
            const boxStaticResource = disk.staticResources.find(v => v.id === resource.resource_ref);

            if (boxStaticResource) {
               boxStaticResources.push({
                  _order: i,
                  mountPoint: resource.mount_point,
                  // eslint-disable-next-line no-underscore-dangle
                  _staticResourceRef: boxStaticResource._ref,
               });
            }
         });
      }

      return boxStaticResources;
   }

   private getWorkloads(
      agentSpec: TPodAgentSpec | undefined,
      podTemplateSpec: TPodTemplateSpec,
      deployUnitSpec: TDeployUnitSpec,
   ) {
      return (agentSpec?.workloads ?? [])
         .filter(w => w.box_ref === this.id)
         .map(w => WorkloadConverter.fromApi(w, podTemplateSpec, deployUnitSpec));
   }
}
